# 
# This program is Copyright (C) 2002 by Marco Roveri <roveri@irst.itc.it>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the Perl Artistic License or the
# GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any
# later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# If you do not have a copy of the GNU General Public License write to
# the Free Software Foundation, Inc., 675 Mass Ave, Cambridge,
# MA 02139, USA.

# The package name
package TTool;

#
# to obtain a prelimiary documentation of the library run
# the command "perldoc TTool.pm".
#
use strict;
use IO::Handle;
use File::stat;

use vars qw($VERSION $VERSION_STRING);
$VERSION        =  1.0;
$VERSION_STRING = "1.0";

use Exporter;
use vars qw(@ISA @EXPORT @EXPORT_OK);
@ISA = qw(Exporter);
BEGIN {
    @EXPORT      = qw( &BMCize &Boundize &CheckCommandExecutable
                       &CheckFname &DoExistsTest &DoFT2IL &DoFTSMV
                       &DoFulfillTest &DoGenericTest &DoIL2SMV
                       &ExtractClasses &ExtractEntity &GenerateSMV
                       &GenExists &GenFulfill &MinusTime &PrintConfig
                       &PrintDiffTime &PrintTime &PrintVersion
                       &RunNuSMV &SetFT2ILExecutable
                       &EnableDirectSMVTranslation &DisableDirectSMVTranslation
                       &SetIL2SMVExecutable &SetNuSMVBMCLength
                       &SetNuSMVCommonOptions &SetNuSMVExecutable
                       &SetNuSMVOrderFile &SetNuSMVSetupFile
                       &SetNuSMVUseBDD &SetNuSMVUseBMC &SetTmpDirectory
                       &SumTime &SetCommandPrefix &CheckNuSMVResults &PrintResults
                       &EnableZchaffUse &DisableZchaffUse &SetCWVerboseTrue &SetCWVerboseFalse
                       &ExtractAndFilterScenario
                     );
}

# 
# Public Variables
#
use vars @EXPORT, @EXPORT_OK;
use vars qw($major_version $minor_version);

#
# Public Subroutines Prototypes
# 
sub BMCize;
sub Boundize;
sub CheckCommandExecutable;
sub CheckFname;
sub DoExistsTest;
sub DoFT2IL;
sub DoFulfillTest;
sub DoGenericTest;
sub DoIL2SMV;
sub DoFT2SMV;
sub ExtractClasses;
sub EnableDirectSMVTranslation;
sub DisableDirectSMVTranslation;
sub EnableZchaffUse;
sub DisableZchaffUse;
sub ExtractEntity;
sub GenerateSMV;
sub GenExists;
sub GenFulfill;
sub MinusTime;
sub PrintConfig;
sub PrintDiffTime;
sub PrintTime;
sub PrintVersion;
sub RunNuSMV;
sub SetFT2ILExecutable;
sub SetIL2SMVExecutable;
sub SetNuSMVBMCLength;
sub SetNuSMVCommonOptions;
sub SetNuSMVExecutable;
sub SetNuSMVOrderFile;
sub SetNuSMVSetupFile;
sub SetNuSMVUseBDD;
sub SetNuSMVUseBMC;
sub SetTmpDirectory;
sub SetCWVerboseTrue;
sub SetCWVerboseFalse;
sub SumTime;
sub SetCommandPrefix;
sub CheckNuSMVResults;
sub PrintResults;

#
# Private Subroutines Prototypes
# 
sub ConfigDefaults ();

#
# Local Variables
# 
# directory where tekmporary files are stored
my $TMP_DIRECTORY     = "tmp";
# the path to the NuSMV executable
my $NUSMV_EXECUTABLE  = "TNuSMV";
# standard NuSMV command line options
my $NUSMV_COMMON_OPTION = "-s -thresh 6000";
# NuSMV variable order file if any
my $NUSMV_ORDER_FILE  = "";
# NuSMV setup script file if any
my $NUSMV_SETUP_FILE  = "";
# Do NuSMV use BDD (0) or SAT (1) for the verification of properties.
my $NUSMV_USES_BMC    = 0; 
# Default path lenght when SAT is used
my $NUSMV_BMC_LENGHT  = 10;
# path to the IL2SMV translator
my $IL2SMV_EXECUTABLE = "il2smv";
# path to the FT2IL translator
my $FT2IL_EXECUTABLE  = "ft2il";
# path to the FT2SMV translator
my $FT2SMV_EXECUTABLE  = "ft2smv";
# MC time limit in seconds
my $TIME_LIMIT = 3600;
# MC memory in 1024 bytes
my $MEMORY_LIMIT = 500000;
# If the unused temporary files must be deleted after their use, set
# to 1. As default they are saved.
my $OPTION_UNLINK_TMP = 0;
# shell command prepend to ft2il, il2smv, NuSMV to retrieve time
# statistic. Actual values are the empty string "" or "time -p".
my $precede_time = "";
# Choose to use the direct translation from FT to SMV or use the intermediate
# explicit intermediate translation to IL.
my $OPTION_DIRECTTO_FT = 1;
# Do we use zchaff or not
my $OPTION_USEZCHAFF = 1;
# Do the counterexample/witness be verbose
my $OPTION_CW_VERBOSE = 1;


###########################################################################
### Library Resident Functions 
###########################################################################

#
# Initializes the local variables of the library.
#
sub ConfigDefaults () {
#    print STDERR "Called ConfigDefaults\n";
    autoflush STDERR, 1;
    autoflush STDOUT, 1;
    $TMP_DIRECTORY       = "tmp";
    $NUSMV_EXECUTABLE    = "TNuSMV";
    $NUSMV_COMMON_OPTION = "-s -thresh 6000";
    $NUSMV_ORDER_FILE    = "";
    $NUSMV_SETUP_FILE    = "";
    $NUSMV_USES_BMC      = 0; 
    $NUSMV_BMC_LENGHT    = 10;
    $IL2SMV_EXECUTABLE   = "il2smv";
    $FT2IL_EXECUTABLE    = "ft2il";
    $TIME_LIMIT          = 10;
    $MEMORY_LIMIT        = 500000;
    $precede_time        = "";
    $OPTION_DIRECTTO_FT  = 1;
    $OPTION_USEZCHAFF    = 1;
    $OPTION_CW_VERBOSE   = 1;
}

###########################################################################
### Library Initialization 
###########################################################################
# 
# Extraction of the Major and Minor version of the Library
#
($major_version, $minor_version) = $VERSION_STRING =~ /^(\d+)\.(\d+)/;

# The library is initialized with default values
ConfigDefaults();

# When loaded the library version is printed on STDOUT
PrintVersion(\*STDOUT);

# Required commands used by the library that needs to be in the
# command search path, i.e. cpp, cp and cat
CheckCommandExecutable("cpp");
CheckCommandExecutable("cp");
CheckCommandExecutable("cat");

###########################################################################
### Library Body 
###########################################################################

# 
# Enable the verbosity of witness/counterexample generated.  That is,
# each state reports for each variable its value
#
sub SetCWVerboseTrue {
    $OPTION_CW_VERBOSE = 1;
}

# 
# Disable the verbosity of witness/counterexample generated.  That is,
# only change of values w.r.t value in the previous state are reported
# in each state
#
sub SetCWVerboseFalse {
    $OPTION_CW_VERBOSE = 0;
}

# 
# Checks if the argument can be executed, i.e. if it can be found in
# the command line search path $PATH environment variable
# 
sub CheckCommandExecutable {
    my $pgm = shift;
    my @d = split(':',$ENV{PATH});
    my $p;

    if ((-e "${pgm}") && (-x "${pgm}")) {
        return 1;
    }
    foreach $p (@d) {
        if ((-e "${p}/${pgm}") && (-x "${p}/${pgm}")) {
            return 1;
        }
    }
    die "ERROR: Unable to locate command \"$pgm\" in PATH\n";
}

# 
# Takes in input an array as returned by times and computes the
# difference among the current return value of times and the given
# argument.
#
sub PrintDiffTime {
    my ($u0,$s0,$cu0,$cs0) = @_;
    my ($u1,$s1,$cu1,$cs1) = times;

    print STDOUT "User time = ", $u1 - $u0, " s\n";
    print STDOUT "System time = ", $s1 - $s0, " s\n";
    print STDOUT "Children User time = ", $cu1 - $cu0, " s\n";
    print STDOUT "Children System time = ", $cs1 - $cs0, " s\n";
}

# 
# Prints a time
#
sub PrintTime {
    my ($u,$s,$cu,$cs,$string) = @_;

    print STDOUT "User time = ", $u, " s\n";
    print STDOUT "System time = ", $s, " s\n";
    print STDOUT "Children User time = ", $cu, " s\n";
    print STDOUT "Children System time = ", $cs, " s\n";
    if ($string eq "") {
        print STDOUT "Total time = ", $u + $s + $cu + $cs, " s\n";
    }
    else {
        print STDOUT "Total ${string} time = ", $u + $s + $cu + $cs, " s\n";
    }
}

# 
# adds two time structures
# 
sub SumTime {
    my @t1;
    my @t2;
    my @re = ();

    ($t1[0], $t1[1], $t1[2], $t1[3], $t2[0], $t2[1], $t2[2], $t2[3]) = @_;

    @re = ($t1[0] + $t2[0], $t1[1] + $t2[1], $t1[2] + $t2[2], $t1[3] + $t2[3]); 
    return(@re);
}

# 
# subtracts two time structures
# 
sub MinusTime {
    my @t1;
    my @t2;
    my @re = ();

    ($t1[0], $t1[1], $t1[2], $t1[3], $t2[0], $t2[1], $t2[2], $t2[3]) = @_;

    @re = ($t1[0] - $t2[0], $t1[1] - $t2[1], $t1[2] - $t2[2], $t1[3] - $t2[3]); 
    return(@re);
}

#
# Prints the Major Minor Version of the Library on the specified file
# descriptor. Call it with PrintVersion(\*STDOUT);
#
sub PrintVersion {
    my $fh = shift;
    print $fh "This is T-Tool library $major_version", ".", "$minor_version\n";
}

#
# Prints the Configuration of the Library
# Call it with PrintConfig(\*STDOUT);
#
sub PrintConfig {
    my $fh = shift;
    print $fh "INFO: ############################################################\n";
    print $fh "INFO: PATH = \"", $ENV{PATH}, "\"\n";
    print $fh "INFO: TMP_DIRECTORY = \"${TMP_DIRECTORY}\"\n";
    print $fh "INFO: NUSMV_EXECUTABLE = \"${NUSMV_EXECUTABLE}\"\n";
    print $fh "INFO: NUSMV_COMMON_OPTION = \"${NUSMV_COMMON_OPTION}\"\n";
    print $fh "INFO: NUSMV_ORDER_FILE = \"${NUSMV_ORDER_FILE}\"\n";
    print $fh "INFO: NUSMV_SETUP_FILE = \"${NUSMV_SETUP_FILE}\"\n";
    print $fh "INFO: NUSMV_USES_BMC = \"${NUSMV_USES_BMC}\"\n";
    print $fh "INFO: NUSMV_BMC_LENGHT = \"${NUSMV_BMC_LENGHT}\"\n";
    print $fh "INFO: IL2SMV_EXECUTABLE = \"${IL2SMV_EXECUTABLE}\"\n";
    print $fh "INFO: FT2IL_EXECUTABLE = \"${FT2IL_EXECUTABLE}\"\n";
    print $fh "INFO: TIME_LIMIT = \"${TIME_LIMIT}\" s\n";
    print $fh "INFO: MEMORY_LIMIT = \"${MEMORY_LIMIT}\" x 1024 bytes\n";
    print $fh "INFO: precede_time = \"${precede_time}\"\n";
    print $fh "INFO: direct transaltion to SMV = \"${OPTION_DIRECTTO_FT}\"\n";
    print $fh "INFO: Counterexample/Witnesses Verbosity = \"${OPTION_CW_VERBOSE}\"\n";
    print $fh "INFO: ############################################################\n";

}

#
# Sets the temporary directory. If the temporary directory does not
# exists returns 1, otherwise 0.
# 
sub SetTmpDirectory {
    my $n = shift;

    if (-d "$n") {
        $TMP_DIRECTORY = $n;
        return 0;
    }
    else {
        print STDERR "ERROR: Directory \"$n\" does not exists.\n";
        return 1;
    }
}

#
# Sets the NuSMV executable. If the empty string is passed returns 1,
# otherwise it returns 0.
# 
sub SetNuSMVExecutable {
    my $n = shift;

    if ("$n" ne "") {
        $NUSMV_EXECUTABLE = $n;
        CheckCommandExecutable($n);
        return 0;
    }
    else {
        print STDERR "ERROR: Cannot set the NuSMV executable to empty string.\n";
        return 1;
    }
}

#
# Sets the prefix command to enable time statistic printing before
# each time consuming computation. Returns 0 unless the command is not
# the empty string or the "time -p" command
# 
sub SetCommandPrefix {
    my $n = shift;

    if (("$n" eq "") || ("$n" eq "time -p")) {
        $precede_time = $n;
        return 0;
    }
    else {
        print STDERR "ERROR: Cannot set the prefix command to anything\n" ,
                     "ERROR: different from empty string or \"time -p\".\n";
        return 1;
    }
}

#
# Sets the NuSMV order file. If the file does not exist returns 1,
# otherwise it returns 0. The empty string is a valid value and
# instructs NuSMV to ignore the order file directive.
# 
sub SetNuSMVOrderFile {
    my $n = shift;

    if (("$n" eq "") || ((-e "$n") && (-r "$n"))) {
        $NUSMV_ORDER_FILE = $n;
        return 0;
    }
    else {
        print STDERR "ERROR: file \"$n\" does not exists or it is not readable.\n";
        return 1;
    }
}

# 
# Sets the NuSMV common options. The empty string is a valid value and
# instructs NuSMV to ignore common options.
# 
sub SetNuSMVCommonOptions {
    my $n = shift;

    $NUSMV_COMMON_OPTION = $n;
    return 0;
}

#
# Sets the NuSMV setup file. If the file does not exist returns 1,
# otherwise it returns 0. The empty string is a valid value and
# instructs NuSMV to ignore the setup file loading.
# 
sub SetNuSMVSetupFile {
    my $n = shift;

    if (("$n" eq "") || ((-e "$n") && (-r "$n"))) {
        $NUSMV_SETUP_FILE = $n;
        return 0;
    }
    else {
        print STDERR "ERROR: file \"$n\" does not exists or it is not readable.\n";
        return 1;
    }
}

#
# Instructs NuSMV to use BMC
#
sub SetNuSMVUseBMC {
    $NUSMV_USES_BMC = 1;
}

#
# Instructs NuSMV to use BDD
#
sub SetNuSMVUseBDD {
    $NUSMV_USES_BMC = 0;
}


#
# Enable direct translation to SMV from FT
#
sub EnableDirectSMVTranslation {
    $OPTION_DIRECTTO_FT = 1;
}

#
# Disable direct translation to SMV from FT
#
sub DisableDirectSMVTranslation {
    $OPTION_DIRECTTO_FT = 0;
}


#
# Enable the use of the zchaff sat solver within NuSMV
#
sub EnableZchaffUse {
    $OPTION_USEZCHAFF = 1;
}

#
# Disable the use of the zchaff sat solver within NuSMV
#
sub DisableZchaffUse {
    $OPTION_USEZCHAFF = 0;
}

#
# Sets the maximum lenght to be used in bounde model checking. If the
# provided length is less or equal to 0, then 1 is returned and an
# error is generated. Otherwise 0 is returned.
# 
sub SetNuSMVBMCLength {
    my $n = shift;

    if ($n > 0) {
        $NUSMV_BMC_LENGHT = $n;
        return 0;
    }
    else {
        print STDERR "ERROR: Cannot assign a length less or equal to 0.\n";
        return 1;
    }
}


#
# Sets the il2smv executable. If the empty string is passed returns 1,
# otherwise it returns 0.
# 
sub SetIL2SMVExecutable {
    my $n = shift;

    if ("$n" ne "") {
        $IL2SMV_EXECUTABLE = $n;
        CheckCommandExecutable($n);
        return 0;
    }
    else {
        print STDERR "ERROR: Cannot set the il2smv executable to empty string.\n";
        return 1;
    }
}

#
# Sets the ft2il executable. If the empty string is passed returns 1,
# otherwise it returns 0.
# 
sub SetFT2ILExecutable {
    my $n = shift;

    if ("$n" ne "") {
        $FT2IL_EXECUTABLE = $n;
        CheckCommandExecutable($n);
        return 0;
    }
    else {
        print STDERR "ERROR: Cannot set the ft2il executable to empty string.\n";
        return 1;
    }
}

    
#
# Checks if the given file name can be opened
# 
sub CheckFname {
    my $fn;
    my $rv = 0;
    ($fn) = @_;
    
    if (-e "$fn") {
        if (-r "$fn") {
            $rv = 0;
        }
        else {
            $rv = 1;
            print STDERR "ERROR: file \"$fn\" is not readable.\n";
        }
    }
    else {
        $rv = 0;
        print STDERR "ERROR: file \"$fn\" does not exists.\n";
    }
    if ($rv == 1) {
        exit(1);
    }
}

#
# Takes in input two file name and an hash. The first file name is the
# IL source file, the second is the destination file. The hash
# associates to each class in the source IL file a bound. This
# function copies the source IL file in to the destination file by
# prefixing the output with the bounds directives specified in the
# bounds hash.
# 
sub Boundize {
    my ($output, %bounds) = @_;
    my $n;

    open(OUT, ">$output") || die "ERROR: Unable to open file $output\n";
    foreach $n (keys %bounds) {
	print OUT "${n} : $bounds{$n}\n";
    }
    print OUT "\n";
    close(OUT) || die "ERROR: Unable to close file $output\n";; 
}
# The below version is relative to the old version of the translators
sub BoundizeOLD {
    my ($input, $output, %bounds) = @_;
    my $n;

    open(OUT, ">$output") || die "ERROR: Unable to open file $output\n";
    foreach $n (keys %bounds) {
	print OUT "BOUND $n : $bounds{$n}\n";
    }
    print OUT "\n";
    close(OUT) || die "ERROR: Unable to close file $output\n";; 

    system("cat $input >> $output") == 0 ||
        die "ERROR: Unable to run cat on $input\n";
}

#
# Takes an FT file and generates an array of classes declared in the
# file. It assumes the file exists and is readable. If the file is not
# readable then it exits with an error.
#
sub ExtractClasses {
    my $file = shift;
    my $cpp = "${TMP_DIRECTORY}/cpp_${file}_cpp";
    my $il  = "${TMP_DIRECTORY}/il_${file}_il";
    my @whatever = ();
 
    if (opendir(D, $TMP_DIRECTORY)) {
        closedir(D);
    }
    else {
        mkdir $TMP_DIRECTORY ||
            die "ERROR: Unable to create \"${TMP_DIRECTORY}\".\n";
    }

    system("cpp -traditional -P < $file > $cpp") == 0 ||
        die "ERROR: Unable to run cpp on file \"$file\".\n";

    DoFT2IL($cpp, $il);

    open(IOFILE, "< ${il}") ||
        die "ERROR: Unable to open file \"$file\"\n";

    while(<IOFILE>) {
        if (/^[\s\t]*CLASS[\s\t]*([a-zA-Z][a-zA-Z0-9_]*)[\s\t]*/) {
            push @whatever, $1;
        }
        next;
    }
    close(IOFILE) || die "ERROR: Unable to close file \"$file\"\n";
    unlink($il) || die "ERROR: Unable to delete file \"$il\"\n";
    unlink($cpp) || die "ERROR: Unable to delete file \"$cpp\"\n";
    return(@whatever);
}


#
# Takes an FT file and generates an array of entities/actors declared in the
# file. It assumes the file exists and is readable. If the file is not
# readable then it exits with an error.
#
sub ExtractEntity {
    my $file = shift;
    my @whatever = ();
    my %tmp;

    open(IOFILE, "< ${file}") ||
        die "ERROR: Unable to open file \"$file\"\n";

    while(<IOFILE>) {
        if (/^[\s\t]*Entity[\s\t]*([a-zA-Z][a-zA-Z0-9_]*)[\s\t]*/) {
            if (!defined($tmp{$1}) ) {
                $tmp{$1} = 1;
                push @whatever, $1;
            }
            else {
                print STDERR "WARNING: Entity $1 already present, ignoring it.\n";
            }
        }
        if (/^[\s\t]*ENTITY[\s\t]*([a-zA-Z][a-zA-Z0-9_]*)[\s\t]*/) {
            if (!defined($tmp{$1}) ) {
                $tmp{$1} = 1;
                push @whatever, $1;
            }
            else {
                print STDERR "WARNING: Entity $1 already present, ignoring it.\n";
            }
        }
        if (/^[\s\t]*entity[\s\t]*([a-zA-Z][a-zA-Z0-9_]*)[\s\t]*/) {
            if (!defined($tmp{$1}) ) {
                $tmp{$1} = 1;
                push @whatever, $1;
            }
            else {
                print STDERR "WARNING: Entity $1 already present, ignoring it.\n";
            }
        }
        if (/^[\s\t]*Actor[\s\t]*([a-zA-Z][a-zA-Z0-9_]*)[\s\t]*/) {
            if (!defined($tmp{$1}) ) {
                $tmp{$1} = 1;
                push @whatever, $1;
            }
            else {
                print STDERR "WARNING: Actor $1 already present, ignoring it.\n";
            }
        }
        if (/^[\s\t]*ACTOR[\s\t]*([a-zA-Z][a-zA-Z0-9_]*)[\s\t]*/) {
            if (!defined($tmp{$1}) ) {
                $tmp{$1} = 1;
                push @whatever, $1;
            }
            else {
                print STDERR "WARNING: Actor $1 already present, ignoring it.\n";
            }
        }
        if (/^[\s\t]*actor[\s\t]*([a-zA-Z][a-zA-Z0-9_]*)[\s\t]*/) {
            if (!defined($tmp{$1}) ) {
                $tmp{$1} = 1;
                push @whatever, $1;
            }
            else {
                print STDERR "WARNING: Actor $1 already present, ignoring it.\n";
            }
        }
        next;
    }
    close(IOFILE) || die "ERROR: Unable to close file \"$file\"\n";
    return(@whatever);
}

#
# Takes two file names in input. The first file specifies an SMV
# file. The second file specifies another SMV file, where each
# occurrence of SPEC LTL_<NUMBER>_SPECF_0 is replaced with 
# LTLSPEC (!LTL_<NUMBER>_SPECF_0 | F 0) to allow for the use of BMC to
# solve the same problem.
# If the $OPTION_UNLINK_TMP variable is set to 1, then the input file
# is removed after the output file is generated.
#
sub BMCize {
    my ($input,$output) = @_;

    open(IN, "<$input");
    open(OUT, ">$output");

    while(<IN>) {
	if (/^SPEC$/) {
	    $_ = <IN>;
	    if (/LTL_([0-9]*)_SPECF_0/) {
		print OUT "LTLSPEC\n";
		print OUT "  (!LTL_${1}_SPECF_0) | F 0\n";
	    }
	    else {
		die "ERROR: BMCize not recognized SPEC";
	    }
	    next;
	}
	print OUT $_;
    }
    close IN;
    close OUT;
    if ($OPTION_UNLINK_TMP == 1) {
        unlink($input);
    }
}

# 
# Generates a formula corresponding to the and of the exists bit of
# each instance of the classes declared in an IL specification. It
# takes in input an hash whose keys are the name of the class and the
# value associated to the key is the number of instances of the
# corresponding class.
# 
sub GenExists {
    my (%bounds) = @_;
    my $formula = "";
    my $i;
    my $c;
    my $n;

    foreach $n (keys %bounds) {
	for ($i = 1; $i <= $bounds{$n}; $i++) {
            # class instances corresponds to the size of arrays.
            if ($formula eq "") {
                $formula = "${n}" . "_${i}.exists";
            }
            else {
                $formula = "$formula & ${n}" . "_${i}.exists";
            }
	}
    }
    return $formula;
}


# 
# Generates a formula corresponding to the and of the fulfillment of
# up to $num instances of the class $cl passed as argument and the
# remaining instances are created but not fulfilled.
# 
sub GenFulfill {
    my ($cl, $num, %bounds) = @_;
    my $formula = "1";
    my $i;
    my $c;

    if (!defined($bounds{$cl})) {
        print STDERR "ERROR: class \"$cl\" has no bound associated.\n";
        exit(1);
    }
    if ($num > $bounds{$cl}) {
        print STDERR "ERROR: $num is greater than the number of instances ",
            "of class\n \"$cl\" (i.e. ", $bounds{$cl}, ")\n";
        exit(1);
    }
    for ($i = 1; $i <= $num; $i++) {
	$formula = "$formula & ${cl}" . "_${i}.exists & ${cl}" . "_${i}.fulfilled";
    }
    for ($i = $num+1; $i <= $bounds{$cl}; $i++) {
	$formula = "$formula & ${cl}" . "_${i}.exists & ! ${cl}" . "_${i}.fulfilled";
    }
    return $formula;
}

# 
# Takes in input a FT file, and the resulting IL file, runs the ft2il
# translator on the FT file and save the result in the IL file. The
# result is the time structure representing the time spent in the
# conversion.
# 
sub DoFT2IL {
    my ($cpp, $il) = @_;
    my @ft2iltime = ();

    printf "*** FT -> IL ***\n";
    CheckCommandExecutable($FT2IL_EXECUTABLE);
    @ft2iltime = times();
    system("${precede_time} ${FT2IL_EXECUTABLE} < $cpp > $il") == 0 ||
        die "ERROR: Unable to run ${FT2IL_EXECUTABLE} on file \"$cpp\"\n";
    @ft2iltime = MinusTime(times(), @ft2iltime);

    return(@ft2iltime);
}

# 
# Takes in input an IL file, and the resulting SMV file, runs the
# il2smv translator on the IL file and save the result in the SMV
# file. The result is the time structure representing the
# time spent in the translation.
# 
sub DoIL2SMV {
    my ($il, $bf, $smv) = @_;
    my @il2smvtime = ();

    printf "*** IL -> SMV ***\n";
    CheckCommandExecutable($IL2SMV_EXECUTABLE);
    CheckCommandExecutable("ltl2smv");

    @il2smvtime = times();
    system("${precede_time} $IL2SMV_EXECUTABLE -bf $bf < $il 2>/dev/null | grep -v -e \"^--\" > $smv") == 0 ||
        die "ERROR: Unable to run $IL2SMV_EXECUTABLE on file ${bf}.\n";
    @il2smvtime = MinusTime(times(), @il2smvtime);
    return(@il2smvtime);
}

# 
# Takes in input a FT file, and the resulting SMV file, runs the ft2smv
# translator on the FT file and save the result in the SMV file. The
# result is the time structure representing the time spent in the
# conversion.
# 
sub DoFT2SMV {
    my ($cpp, $bf, $ft) = @_;
    my @ft2smvtime = ();

    printf "*** FT -> SMV ***\n";
    CheckCommandExecutable($FT2SMV_EXECUTABLE);
    @ft2smvtime = times();
    system("${precede_time} ${FT2SMV_EXECUTABLE} -bf $bf < $cpp > $ft") == 0 ||
        die "ERROR: Unable to run ${FT2SMV_EXECUTABLE} on file \"$cpp\"\n";
    @ft2smvtime = MinusTime(times(), @ft2smvtime);

    return(@ft2smvtime);
}

# 
# Takes in input a FT file, a test name, a name to be used as defined
# symbol to provide to the preprocessor to extract the given test and
# an hash with the bounds. It returns the time structure representing
# the time spent in the routine.
# 
sub GenerateSMV {
    my ($file, $testname, $test, %bounds) = @_;
    my @ft2iltime;
    my @il2smvtime;
    my @re;
    my $ft = "${TMP_DIRECTORY}/${testname}.sft";
    my $cpp = "${TMP_DIRECTORY}/${testname}.ft";
    my $il = "${TMP_DIRECTORY}/${testname}.il";
    my $boundfile = "${TMP_DIRECTORY}/${testname}.bil";
    my $smv = "${TMP_DIRECTORY}/${testname}.smv";
    my $bsmv = "${TMP_DIRECTORY}/${testname}.bsmv";
    my $smvfile = "";
    my $ilfile = "";
    my $cppdef = "";

    if (opendir(D, $TMP_DIRECTORY)) {
        closedir(D);
    }
    else {
        mkdir $TMP_DIRECTORY ||
            die "ERROR: Unable to create \"${TMP_DIRECTORY}\".\n";
    }

    system("cp $file $ft") == 0 ||
        die "ERROR: Unable to copy \"$file\" to \"$ft\".\n";
    if ($test ne "") {
        $cppdef = "-D${test}";
    }

    system("cpp -traditional -P ${cppdef} < $ft > $cpp") == 0 ||
        die "ERROR: Unable to run cpp on file \"$ft\".\n";

    if ($OPTION_DIRECTTO_FT == 1) {
        # Creates bounds file
        Boundize($boundfile, %bounds);

        # Converting the FT file to IL 
        @re = DoFT2SMV($cpp, $boundfile, $smv);
        print STDOUT "################# FT -> SMV time #################\n";
        PrintTime(@re, "ft2smv");
        print STDOUT "##################################################\n";
    }
    else {
        # Converting the FT file to IL 
        @ft2iltime = DoFT2IL($cpp, $il);
        print STDOUT "################# FT -> IL time ##################\n";
        PrintTime(@ft2iltime, "ft2il");
        print STDOUT "##################################################\n";

        # Creates bounds file
        Boundize($boundfile, %bounds);

        # Converting the IL file to SMV
        @il2smvtime = DoIL2SMV($il, $boundfile, $smv);
        print STDOUT "################# IL -> SMV time ##################\n";
        PrintTime(@il2smvtime, "il2smv");
        print STDOUT "###################################################\n";
        @re = SumTime(@il2smvtime, @ft2iltime);
        $ilfile = $il;
    }

    if ($NUSMV_USES_BMC == 1) {
        BMCize($smv, $bsmv);
        $smvfile = $bsmv;
    }
    else {
        $smvfile = $smv;
    }
    return((@re , $smvfile, "$ilfile"));
}

# 
# Run the NuSMV model checker on the specified SMV file. If a setup
# file has been provided and an ordering file is provided they are
# used. Depending on the verification technology used different
# options are passed to the model checker to respect the choice.
# It returns the time spent in verifying the given property.
# 
sub RunNuSMV {
    my $smvfile = shift;
    my @q = times;
    my $cmd = "";
    my $hml = $MEMORY_LIMIT + 5000;
    my $precmd = "ulimit -t $TIME_LIMIT; ulimit -d $MEMORY_LIMIT; ulimit -m $hml;";
    my $smvcmd = "${NUSMV_EXECUTABLE} ${NUSMV_COMMON_OPTION} -int";
    my $smvscript = "";
    my $errfile = "${smvfile}.err";
    my $outfile = "${smvfile}.out";
    my @re = ();

    if ($NUSMV_USES_BMC == 1) {
        if ($OPTION_USEZCHAFF == 1) {
            $smvscript = "echo go_bmc; echo time; echo set sat_solver zchaff; echo check_ltlspec_bmc; echo time; echo print_usage; echo quit";
        }
        else {
            $smvscript = "echo go_bmc; echo time; echo check_ltlspec_bmc; echo time; echo print_usage; echo quit";
        }
        $smvcmd =~ " -bmc -bmc_length ${NUSMV_BMC_LENGHT} ";
    }
    else {
        $smvscript = "echo go; echo time; echo check_spec; echo time; echo print_usage; echo quit";
        $smvcmd =~ " ";
    }
    if ($NUSMV_SETUP_FILE ne "") {
        $smvcmd =~ " -load $NUSMV_SETUP_FILE ";
    }
    if ($NUSMV_ORDER_FILE ne "") {
        $smvcmd =~ " -i $NUSMV_ORDER_FILE ";
    }
        
    printf "*** RUN NuSMV ***\n";

    CheckCommandExecutable($NUSMV_EXECUTABLE);
    $cmd = "$precmd ($smvscript) | ${precede_time} $smvcmd $smvfile 2>${errfile} | grep -v \"LTL_\" >${outfile}";

    @q = times;
    system($cmd) == 0 ||
        die "ERROR: Unable to execute NuSMV on file \"$smvfile\"\n";
    @q = MinusTime(times(), @q);

    # Process results
    @re = CheckNuSMVResults($outfile, @q);

    print STDOUT "################# SMV time #################\n";
    PrintTime(@q, "NuSMV");
    print STDOUT "############################################\n";
    return(@re);
}

# 
# Extracts informations from the output produced by
# NuSMV. The resulting structure contains:
# re[0]  = User time;
# re[1]  = System time;
# re[2]  = Children user time;
# re[3]  = Children system time;
# re[4]  = Max num of pb not solved with BMC (-1 if no BMC)
# re[5]  = Status of the property (1 = true, 0 = false, -1 don't know)
# re[6]  = Counterexample/Witness lenght if any (-1 if no counterexample)
# re[7]  = User time as provided by NuSMV
# re[8]  = System time as provided by NuSMV
# re[9]  = SBRK memory by NuSMV
# re[10] = Initialized memory by NuSMV
# re[11] = Uninitialized Memory by NuSMV
# re[12] = The output file
# 
sub CheckNuSMVResults {
    my ($file, @t) = @_;
    my $nobugat = -1;
    my $isfalse = -1;
    my $clenght = -1;
    my $ctrace  = -1;
    my $usertime = -1;
    my $systemtime = -1;
    my $sbrkmem = -1;
    my $initmem = -1;
    my $uninitmem = -1;

    open(DATA, "$file") || die "ERROR: Unable to open result file \"$file\"\n";
    while(<DATA>) {
        chomp;

        # We do not print the SMV output
        # print STDOUT "$_\n";
        if (/--\sno\scounterexample\sfound\swith\sbound\s([0-9]+)\s/) {
            $nobugat = $1;
            next;
        }
        if (/--\sspecification\s.+\s+is\sfalse$/) {
            $isfalse = 1;
            next;
        }
        if (/--\sspecification\s.+\s+is\strue$/) {
            $isfalse = 0;
            next;
        }
        if (/->\sState\s([0-9]+).([0-9]+)\s<-$/) {
            $ctrace = $1 if ($ctrace == -1);
            if (($ctrace == $1) && ($clenght < $2)) {
                $clenght = $2;
            }
            else {
                print STDERR "CheckNuSMVResults: something wrong\n";
                exit 1;
            }
            next;
        }
        if (/User\stime\s+([0-9.]+)\sseconds$/) {
            $usertime = $1;
            next;
        }
        if (/System\stime\s+([0-9.]+)\sseconds$/) {
            $systemtime = $1;
            next;
        }
        if (/\s+data\ssize\ssbrk\s+=\s+([0-9]+)K$/) {
            $sbrkmem = $1;
            next;
        }
        if (/\s+data\ssize\sinitialized\s+=\s+([0-9]+)K$/) {
            $initmem = $1;
            next;
        }
        if (/\s+data\ssize\suninitialized\s+=\s+([0-9]+)K$/) {
            $uninitmem = $1;
            next;
        }
    }
    close(DATA);
    return(($t[0], $t[1], $t[2], $t[3], $nobugat, $isfalse, $clenght, $usertime, $systemtime, $sbrkmem, $initmem, $uninitmem, $file));
}

#
# Extracts the counterexample/witness generated by NuSMV and filters
# it removing for each "class" the printing of the class attributes if
# classname.exists is not true. Moreover, depending on the second
# argument, the printed scenario shows in each state only the
# variables that changed value w.r.t. the corresponding value in the
# previous state.
#
sub ExtractAndFilterScenario {
    my ($file,$verb) = @_;
    my $state=0;
    my $trace= 0;
    my %loop;
    my %variables;
    my %classes;

    my $n;
    my $i;
    my $s;

    open(FILE, "<$file") || die "ERROR: Unable to open file $file\n";
    while(<FILE>) {
        my $l = chop($_);
        if (/^->\sState\s([0-9]+).([0-9]+)\s<-$/) {
            $trace = $1;
            $state++;
            next;
        }
        if (/^--\sloop\sstarts\shere\s--$/) {
            $loop{$state+1} = 1;
            next;
        }
        if (/^\s+([A-Za-z_.0-9\[\]]+)\s=\s([A-Za-z0-9]+)$/) {
            if (!defined($variables{$1})) {
                $variables{$1} = {};
            }
            $variables{$1}{$state} = $2;
            next;
        }
    }

    # Fill the assignement to state variables if scenario reports
    # only changes w.r.t. value of a variable in the previous state.
    foreach $n (keys %variables) {
        my $val = "*";
        for ($i = 1; $i <= $state; $i++) {
            if (not defined($variables{$n}{$i}) ) {
                $variables{$n}{$i} = $val;;
            }
            else {
                $val = $variables{$n}{$i};
            }
        }
    }

    # Creating class structure
    foreach $n (keys %variables) {
        my $root;
        my $el;

        $_ = $n;

        # Strong assumption that the varaibles are "class.attribute"
        # otherwise the regexp below must be changed
        if (/([A-Za-z_0-9\[\]]+).([A-Za-z_0-9]+)/) {
            $root = $1;
            $el = $2;
        }
        else {
            die "ERROR: wrong regexp for representing varaible assingment\n";
        }
        $classes{$root}{$el} = $variables{$n};
    }

    # Printing trace considering changes depending on value of 
    # $verb specified as input.
    for ($i = 1; $i <= $state; $i++) {
        if (defined($loop{$i})) {
            print "-- loop starts here --\n";
        }
        print "-> State ${trace}.${i} <-\n";
        foreach $n (keys %classes) {
            foreach $s (keys %{$classes{$n}}) {
                if (defined($classes{$n}{"exists"}) &&
                    ($classes{$n}{"exists"}{$i} == 1)) {
                    if (($i == 1) || ($verb == 1)) {
                        print "    ${n}.${s} = $classes{$n}{$s}{$i}\n";
                    }
                    elsif (not ($classes{$n}{$s}{$i} == $classes{$n}{$s}{$i-1})) {
                        print "    ${n}.${s} = $classes{$n}{$s}{$i}\n";
                    }
                }
            }
        }
    }
}

# 
# Prints the result of the call to NuSMV into a uman "readable" form
# 
sub PrintResults {
    my ($smvfile, $ilfile, $name, $ut, $st, $cut, $cst, $nobugat, $isfalse, $clenght, $usertime, $systemtime, $sbrkmem, $initmem, $uninitmem, $outsmvfile) = @_;
    my $propstatus = "Unrecognized";

    $propstatus = "CTX/Witn" if ($isfalse == 1);
    $propstatus = "No CTX/Witn" if ($isfalse == 0);

    printf "##################################################\n";
    printf "DATALINE %s | %10s | %10s | %10s | %10s | %10s | %10s | %10s | %10s | %10s",
            $name, $ut + $cut + $st + $cst, $propstatus, $clenght, $nobugat,
            $usertime + $systemtime, $sbrkmem, $initmem, $uninitmem, stat($smvfile)->size;
    if ($OPTION_DIRECTTO_FT == 0) {
        printf " | %10s", stat($ilfile)->size;
    }
    printf " |\n";
    printf "##################################################\n";
    printf "DATA %s: Time = %s sec\n", $name, $ut + $cut + $st + $cst;
    printf "DATA %s: Property status = %s\n", $name, $propstatus;
    printf "DATA %s: Counterexample length = %s\n", $name, $clenght;
    printf "DATA %s: No Bug at length = %s\n", $name, $nobugat;
    printf "DATA %s: Time (NuSMV) = %s sec\n", $name, $usertime + $systemtime;
    printf "DATA %s: SBRK mem = %s Kb\n", $name, $sbrkmem;
    printf "DATA %s: Initialized mem = %s Kb\n", $name, $initmem;
    printf "DATA %s: Uniinitialized mem = %s Kb\n", $name, $uninitmem;
    printf "DATA %s: Size SMV file = %s b\n", $name, stat($smvfile)->size;
    if ($OPTION_DIRECTTO_FT == 0) {
        printf "DATA %s: Size IL file = %s b\n", $name, stat($ilfile)->size;
    }
    printf "##################################################\n";
    if ($isfalse == 1) {
        printf "### The scenario extracted is the following\n";
        printf "##################################################\n";
        ExtractAndFilterScenario("${outsmvfile}", $OPTION_CW_VERBOSE);
        printf "##################################################\n";
    }
}


# 
# Performs a generic test on a FT specification. It takes as argument
# the test name, the FT file, the macro name used in the FT file to
# identify the current test, and the bounds array.
# 
sub DoGenericTest {
    my ($testname, $file, $test, %bounds) = @_;
    my @time1;
    my @res;
    my @tottime;
    my $smvfile;
    my $ilfile;

    # replace spaces with underscore
    $testname =~ tr/\ /_/s;

    print "******************** Test $testname ********************\n";
    ($time1[0], $time1[1], $time1[2], $time1[3], $smvfile, $ilfile) =
        GenerateSMV($file, $testname, $test, %bounds);

    @res = RunNuSMV($smvfile);
    @tottime = SumTime(($res[0], $res[1], $res[2], $res[3]), @time1);
    print STDOUT "################# $testname time #################\n";
    PrintTime(@tottime, "test");
    print STDOUT "##################################################\n";
    # Storing the new time struct so far in res
    $res[0] = $tottime[0];
    $res[1] = $tottime[1];
    $res[3] = $tottime[2];
    $res[3] = $tottime[3];
    PrintResults($smvfile, "$ilfile", $testname, @res);
}

# 
# Performs the Exists consistency check. It takes as argument the
# test name, the FT file, and the bounds for the current problem.
# 
sub DoExistsTest {
    my ($testname, $file, %bounds) = @_;
    my @time1;
    my @res;
    my @tottime;
    my $smvfile;
    my $ilfile;
    my $formula = "";

    # replace spaces with underscore
    $testname =~ tr/\ /_/s;

    print "******************** Test $testname ********************\n";
    ($time1[0], $time1[1], $time1[2], $time1[3], $smvfile, $ilfile) =
        GenerateSMV($file, $testname, "", %bounds);

    $formula = GenExists(%bounds);

    open(OUT, ">>$smvfile") || die "ERROR: Unable to open \"$smvfile\"\n";
    if ($NUSMV_USES_BMC == 1) {
        print OUT "LTLSPEC ! F ($formula & G 1)\n";
    }
    else {
        print OUT "SPEC ! EF ($formula & EG 1)\n";
    }
    close OUT || die "ERROR: Unable to close \"$smvfile\"\n";;

    @res = RunNuSMV($smvfile);

    @tottime = SumTime(($res[0], $res[1], $res[2], $res[3]), @time1);
    print STDOUT "################# $testname time #################\n";
    PrintTime(@tottime, "test");
    print STDOUT "##################################################\n";
    # Storing the new time struct so far in res
    $res[0] = $tottime[0];
    $res[1] = $tottime[1];
    $res[3] = $tottime[2];
    $res[3] = $tottime[3];
    PrintResults($smvfile, "$ilfile", $testname, @res);
}


# 
# Performs the Fulfill consistency check. It takes as argument the
# test name, the FT file, the name of a class, the upper limit for
# number of fulfilled class instances and the bounds for the current
# problem.
# 
sub DoFulfillTest {
    my ($testname, $file, $dep, $num, %bounds) = @_;
    my @time1;
    my @res;
    my @tottime;
    my $smvfile;
    my $ilfile;
    my $formula = "";

    # replace spaces with underscore
    $testname =~ tr/\ /_/s;

    print "******************** Test $testname ********************\n";
    ($time1[0], $time1[1], $time1[2], $time1[3], $smvfile, $ilfile) =
        GenerateSMV($file, $testname, "", %bounds);

    $formula = GenFulfill($dep, $num, %bounds);

    open(OUT,">>$smvfile") || die "ERROR: Unable to open \"$smvfile\"\n";
    if ($NUSMV_USES_BMC == 1) {
        print OUT "LTLSPEC ! F G ($formula)\n";
    }
    else {
        print OUT "SPEC ! EF EG ($formula)\n";
    }
    close OUT || die "ERROR: Unable to close \"$smvfile\"\n";;

    @res = RunNuSMV($smvfile);

    @tottime = SumTime(($res[0], $res[1], $res[2], $res[3]), @time1);
    print STDOUT "################# $testname time #################\n";
    PrintTime(@tottime, "test");
    print STDOUT "##################################################\n";
    # Storing the new time struct so far in res
    $res[0] = $tottime[0];
    $res[1] = $tottime[1];
    $res[3] = $tottime[2];
    $res[3] = $tottime[3];
    PrintResults($smvfile, "$ilfile", $testname, @res);
}

1;

__END__

#
# WARNING: do not remove the above empty line.
# 

###########################################################################
### Library Documentation
###########################################################################
=pod

=head1 NAME

TTool - The T-Tool perl library

=head1 SYNOPSIS;

use TTool;

=head1 DESCRIPTION

The C<TTool> library provides a simple mechanism to perform checking
of FT specifications. It provides the user with high level functions
and low level functions for exper users to write your own
checking. The list of functions exported by the library is the
following where functions have been grouped depending their role.

=head2 Generic High level test functions

=over      

=item DoExistsTest($testname, $file, %bounds)

Performs the Exists consistency check. It takes as argument the test
name C<$testname>, the FT file C<$file>, and the bounds hash
C<%bounds> specifying the number of instances per classfor the current
problem. It prints on STDOUT the total amount of time required to
perform the verification (i.e. translation time to il, translation
time to SMV and verification time of the SMV problem).

=item DoFulfillTest($testname, $file, $cl, $num, %bounds)

Performs the Fulfill consistency check. It takes as argument the test
name C<$testname>, the FT file C<$file>, the name of a class C<$cl>,
the upper limit C<$num> for the number of class instances of the
considered class (see C<GenFulfill> function) and an hash C<%bounds>
specifying the number of instances per classfor the current problem.

=item DoGenericTest($testname, $file, $test, %bounds)

Performs a generic test on a FT specification. It takes as argument
the test name C<$testname>, the FT file C<$file>, the macro name
C<$test> used in the FT file to identify the current test, and a
bounds hash C<%bounds> specifying the number of instances per
class. It prints on STDOUT the total amount of time required to
perform the verification (i.e. translation time to il, translation
time to SMV and verification time of the SMV problem).

=back

=head2 Generic intermediate level functions

=over      

=item DoFT2IL($file, $il)

Takes in input a FT file C<$file>, and the resulting IL file C<$il>,
runs the ft2il translator on the FT file and save the result in the IL
file. The result is the time structure representing the time spent in
the conversion.


=item DoIL2SMV($file, $bfile, $smv)

Takes in input an IL file C<$file>, the bound file
C<$bfile> and the resulting SMV file C<$smv>, runs the
il2smv translator on the IL file and save the result in the
SMV file. The result is the time structure representing the
time spent in the translation.


=item DoFT2SMV($file, $bfile, $smv)

Takes in input an FT file C<$file>, the bound file
C<$bfile> and the resulting SMV C<$smv> file, runs the
ft2smv translator on the FT file and save the result in the
SMV file. The result is the time structure representing the
time spent in the conversion.


=item GenerateSMV($file, $testname, $test, %bounds)

Takes in input a FT file C<$file>, a test name
C<$testname>, a name C<$test> to be used as defined symbol
to provide to the preprocessor to extract the corresponding
test and an hash C<%bounds> with the bounds on the number
of instances for each declared class. It returns the time
structure representing the time spent in the routine.  If
the C<EnableDirectSMVTranslation()> option has been chosen,
then the direct translation to of an FT specification to
SMV is performed. Otherwise, the IL version of the
specification is generated.


=item RunNuSMV($file)

Run the NuSMV model checker on the specified SMV file
C<$file>. If a setup file has been provided with the
C<SetNuSMVSetupFile()> function and an ordering file is
provided with the C<SetNuSMVOrderFile()> they are
used. Depending on the verification technology used
different options are passed to the model checker to
respect the choice. It returns the time structure
representing the time spent in verifying the given
property. If the C<SetCommandPrefix()> is specified the
execution of the command prints also additional statistic
on the standard error file. If the ZChaff SAT solver has
been selected, while doing BMC it will be used. The results
of the verification are stored in C<$file.out> and an array
representing the result of the verification is returned for
pretty printing via C<PrintResults()>.


=item BMCize($sfile, $dfile)

Takes two file names in input. The first file C<$sfile> specifies an
SMV file. The second file C<$dfile> specifies another SMV file, where
each occurrence of 

C<SPEC LTL_<NUMBER>_SPECF_0> 

is replaced with

C<LTLSPEC (!LTL_E<lt>NUMBERE<gt>_SPECF_0 | F 0)> 

to allow for the use of BMC to solve the same problem. If the
C<$OPTION_UNLINK_TMP> variable is set to C<1>, then the input file is
removed after the output file is generated.  


=item Boundize($bfile, %bounds)

Takes in input the file (C<$file>) where to store class
bounds specified in the hash C<%bounds>.

=back

=head2 Generic utility routines

=over

=item CheckCommandExecutable($file)

Checks whether the argument C<$file> refers to a file that can be
executed, i.e. if it can be found in the command line search path
C<$PATH> environment variable.

=item CheckFname($file)

Checks whether the C<$file> exists and is readable. If it is not the
case it exits with an error.

=item ExtractClasses($ftfile)

Extracts from an FT C<$ftfile> an array representing the classes
resulting from the translation into IL. It assumes the file exists and
is readable. If the file is not readable then it exits with an error.

=item ExtractEntity($ftfile)

Extracts from an FT C<$ftfile> an array representing the entities/actors
declared in the FT file. It assumes the file exists and is
readable. If the file is not readable then it exits with an error.

=item GenExists(%bounds)

Generates a formula corresponding to the and of the exists bit of each
instance of the classes declared in an IL specification. It takes in
input an hash C<%bounds> whose keys are the name of the classes and
the value associated to the key is the number of instances of the
corresponding class.

=item GenFulfill($cl, $num, %bounds)

Generates a formula corresponding to the conjunction of the
fulfillment of up to C<$num> instances of the class C<$cl> passed as
argument and the remaining C<$bounds{$cl} - $num> instances are created
but are not yet fulfilled.

=item CheckNuSMVResults($outfile, @time)

Extracts informations from the output produced by NuSMV and
stored in file C<$file>. C<@time> is the time structure
corresponding to the call to NuSMV. The resulting structure
contains:

=over 4

=item re[0] 

User time;

=item re[1]

System time;

=item re[2]

Children user time;

=item re[3]

Children system time;

=item re[4]

Max num of pb not solved with BMC (-1 if no BMC)

=item re[5]

Status of the property (1 = true, 0 = false, -1 don't know)

=item re[6]

Counterexample/Witness lenght if any (-1 if no counterexample)

=item re[7]

User time as provided by NuSMV

=item re[8]

System time as provided by NuSMV

=item re[9]

SBRK memory by NuSMV

=item re[10]

Initialized memory by NuSMV

=item re[11]

Uninitialized Memory by NuSMV

=back


=item ExtractAndFilterScenario($smfile, $verbosity)

Extracts the counterexample/witness generated by NuSMV and filters it
removing for each "class" the printing of the class attributes if
classname.exists is not true. Moreover, depending on the second
argument, the printed scenario shows in each state only the variables
that changed value w.r.t. the corresponding value in the previous
state. If C<$verbosity> is 1 (default), then for each state the value
of all the variables is printed. If it is 0, only the value of
variables that changed value w.r.t. previous state are printed.


=item PrintResults($smfile, $ilfile, $name, @result)

Prints the results of a call to NuSMV via C<RunNuSMV()> into a uman
"readable" form. Moreover some statistical infos (i.e., the file size)
of C<$smvfile> and C<$ilfile> are printed out. The output depends on
the value of C<$OPTION_DIRECTTO_FT> (if set to one, then the stat of
file C<$ilfile> are printed out, otherwise no info is printed for this
file).

=back

=head2 Time handling routines

Time is a structure represented as an array of four elements
C<(u,s,cu,cs)>, where C<u> and C<s> are the user and system time
respectively of the current process, C<cu> and C<cs> are the user and
system time respectively of the child process. This structure is
according to the result of the C<time()> function.

=over


=item PrintDiffTime(@lt)

Takes in input an array C<@lt> representing a time structure and
computes the difference among the current return value of times and
the given argument and prints the result of the difference on STDOUT.


=item PrintTime(@lt [,string])

Takes in input an array C<@lt> representing a time structure and an
optional string. Prints the all the elements of the time structure on
STDOUT and computes and prints the sum of the four element times. The
C<string> argument will appear in the total time printing as

    "Total ${string} time = 10 s"

Thus making it easy the retrieval via automatic mechanism of the total
time spent in performing a given task.


=item SumTime(@t1, @t2)

Computes and returns a time structure resulting by adding time
elements postion by position.


=item MinusTime(@t1, @t2)

Computes and returns a time structure resulting by subtracting time
elements postion by position.

=back

=head2 Options handling functions

=over

=item PrintVersion(FILE)

Prints the librbary version on FILE.


=item PrintConfig(FILE)

Prints the librbary current configuration on FILE.


=item EnableDirectSMVTranslation

Enable direct translation to SMV from FT


=item DisableDirectSMVTranslation

Disable direct translation to SMV from FT


=item EnableZchaffUse

Enable the use of the zchaff sat solver within NuSMV


=item DisableZchaffUse

Disable the use of the zchaff sat solver within NuSMV


=item SetCWVerboseTrue

Enable the verbosity of witness/counterexample generated. That is,
each state reports for each variable its value


=item SetCWVerboseFalse

Disable the verbosity of witness/counterexample generated.  That is,
only change of values w.r.t value in the previous state are reported
in each state


=item SetCommandPrefix(string)

Sets the prefix command to enable time statistic
printing before each time consuming computation.
Returns 0 unless the cmdstring is not the empty
string or the "time -p" command


=item SetFT2ILExecutable(string)

Sets the FT2IL executable. If the empty string is
passed returns 1, otherwise it returns 0.


=item SetIL2SMVExecutable(string)

Sets the IL2SMV executable. If the empty string is
passed returns 1, otherwise it returns 0.


=item SetNuSMVBMCLength(int)

Sets the maximum lenght to be used in bounde model
checking. If the provided length is less or equal
to 0, then 1 is returned and an error is generated.
Otherwise 0 is returned.


=item SetNuSMVCommonOptions(string)

Sets the NuSMV common options (i.e. default command line
options). The empty string is a valid value and instructs
NuSMV to ignore common options.


=item SetNuSMVExecutable(string)

Sets the NuSMV executable. If the empty string is
passed returns 1, otherwise it returns 0.


=item SetNuSMVOrderFile(string)

Sets the NuSMV order file. If the file does not exist
returns 1, otherwise it returns 0. The empty string
is a valid value and instructs NuSMV to ignore the
order file directive.


=item SetNuSMVSetupFile(string)

Sets the NuSMV setup file. If the file does not exist
returns 1, otherwise it returns 0. The empty string is
a valid value and instructs NuSMV to ignore the loading of
the setup file. The file must contain valis NuSMV
interactive commands.


=item SetNuSMVUseBDD(Z<>)

Instructs NuSMV to use BDDs.


=item SetNuSMVUseBMC(Z<>)

Instructs NuSMV to use SAT.


=item SetTmpDirectory(string)

Sets the temporary directory where temporary files are
saved. If the temporary directory does not exists
returns 1, otherwise 0 is returned.

=back

=head1 EXAMPLE


    #! /usr/bin/perl -I /opt/software/T-Tool/lib
    #
    # Assuming that TTool.pm perl module has been saved
    # in directory /opt/software/T-Tool/lib

    use TTool;
        
    sub bound1_test {
        my ($tstfile, $c, $e) = @_;
        my @classes = @$c;
        my @entity = @$e;
        my %bounds1;
        my %isentity;
        my $cl;
     
        foreach $cl (@entity) {
          $isentity{$cl} = 1;
        }
        print "#####################################################\n";
        print "CHECKS FOR BOUND 1\n";
        print "#####################################################\n";
        foreach $cl (@classes) {
            $bounds1{$cl} = 1;
        }
    
        # exists the possibility to create all the instances
        DoExistsTest("EXISTS_1", "$tstfile", %bounds1);
    
        foreach $cl (@classes) {
            if (defined($isentity{$cl})) {
                print "Skipping Class \"$cl\" since it is an Actor/Entity\n";
            }
            else {
                DoFulfillTest("FULFILL_${cl}_0_1", "$tstfile", "$cl", 0, %bounds1);
                DoFulfillTest("FULFILL_${cl}_1_1", "$tstfile", "$cl", 1, %bounds1);
            }
        }
    
        for ($i = 1; $i <= 12; $i++) {
            DoGenericTest("ASSERTION_${i}_1", "$tstfile", "ASSE_${i}", %bounds1);
        }
        print "#####################################################\n";
    }
    
    
    sub main {
        my @classes = ();
        my $tstfile = "";
        my @entity = ();
    
        if ($#ARGV == 0) {
            $tstfile = $ARGV[0];
        }
        else {
            print "WARNING: No file specified. Exiting.\n";
            exit(1);
        }
    
        SetNuSMVUseBMC();
    
        PrintConfig(\*STDOUT);
    
        CheckFname($tstfile);
    
        @entity = ExtractEntity($tstfile);
    
        @classes = ExtractClasses($tstfile);
    
        bound1_test($tstfile, \@classes, \@entity);
    }
    
    main();

=head1 AUTHOR

Marco Roveri E<lt>roveri@irst.itc.itE<gt>

=head1 COPYRIGHT

Copyright (c) 2002-2003  Marco Roveri E<lt>roveri@irst.itc.itE<gt>

This program is free software; you can redistribute it and/or modify
it under the terms of the Perl Artistic License or the GNU General
Public License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

If you do not have a copy of the GNU General Public License write to
the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
USA.

=head1 SEE ALSO

=over 4

=item The T-Tool Home Page

http://sra.itc.it/tools/t-tool

=item The NuSMV Model Checker Home Page

http://nusmv.irst.itc.it

=back

=head1 VERSION

T-Tool library 1.0, January 2003.

=cut

