/******************************************************************************

  FileName    [ft-bounds.c]

  PackageName [ft]

  Synopsis    [The bounds manager for the FT to IL translation]

  Author      [Marco Pistore] 

  Copyright   [Copyright (C) 2003 by University of Trento.

  T-Tool is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  T-Tool is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information on the T-Tool see <http://dit.unitn.it/~ft>
  or email to <ft@dit.unitn.it>. Please report bugs to <ft@dit.unitn.it>.]

******************************************************************************/

#include <malloc.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <regex.h>
#include "ft.h"

struct _tBounds {
  tName name;
  int bound;
  tBounds next;
};

tBounds mkEmptyBounds()
{
  return NULL;
}

tBounds addBound(tBounds bs, tName n, int b)
{
  tBounds bs_new = ALLOC(struct _tBounds);
  bs_new->name = n;
  bs_new->bound = b;
  bs_new->next = bs;
  return bs_new;
}

tBounds zeroBounds(tDom dom)
{
  tBounds bounds = mkEmptyBounds();

  tElementSet els = getDomElements(dom);

  while (! isemptyElementSet(els)) {
    tElement el = firstElement(els);

    bounds = addBound(bounds, getElementName(el), 0);

    els = nextElementSet(els);
  }

  return bounds;
}

tBounds singleBounds(tDom dom)
{
  tBounds bounds = mkEmptyBounds();

  tElementSet els = getDomElements(dom);

  while (! isemptyElementSet(els)) {
    tElement el = firstElement(els);

    bounds = addBound(bounds, getElementName(el), 1);

    els = nextElementSet(els);
  }

  return bounds;
}

tBounds doubleBounds(tDom dom)
{
  tBounds bounds = mkEmptyBounds();

  tElementSet els = getDomElements(dom);

  while (! isemptyElementSet(els)) {
    tElement el = firstElement(els);

    bounds = addBound(bounds, getElementName(el), 2);

    els = nextElementSet(els);
  }

  return bounds;
}

tBounds parseBounds(FILE * f, tDom dom)
{
  tBounds bs = zeroBounds(dom);

  while(1) {
    tBounds tmp_bs = bs;
    int res;
    regex_t rec;
    char * p;
    char * pattern;
    int b;

    res = fscanf(f, "%as : %d\n", &p, &b);
    assert(res == EOF || res == 2);
    if (res == EOF) break;

    pattern = malloc(strlen(p)+2+1);

    sprintf(pattern,"^%s$", p);

    res = regcomp(&rec, pattern, 0);
    assert(res == 0);

    while(tmp_bs) {
      if (regexec(&rec, getNameString(tmp_bs->name), 0, NULL, 0) == 0) {
	tmp_bs->bound = b;
      }
      tmp_bs = tmp_bs->next;
    }
  }

  return bs;
}

tDom instantiateDomain(tDom dom, tBounds bounds)
{
  tDom domNew = mkEmptyDom();

  tPropSet props = getDomGlobalProps(dom);

  for (; bounds; bounds = bounds->next) {
    tName n = bounds->name;
    int b = bounds->bound;
    
    tElement el = getDomElementByName(dom, n);

    assert(el);
    assert(b >= 0);

    if (b == 0) {
      fprintf(stderr, "INFO: empty range for element %s.\n",
	      getNameString(getElementName(el)));
      continue;
    }

    {
      tNameSet range = mkNoNames();
      while (b > 0) {
	char * str;

	str = malloc(strlen(getNameString(getElementName(el))) + 10 + 2);
	assert(str);
	sprintf(str, "%s_%d", getNameString(getElementName(el)), b);
	range = mkConsNameSet(mkName(str),range);

	b--;
      }

      setElementRange(el, range);
      domNew = addElement(domNew, el);
    }

  }

  while (! isemptyPropSet(props)) {
    tProp prop = firstProp(props);

    domNew = addGlobalProp(domNew, prop);

    props = nextPropSet(props);
  }
  
  return domNew;
}
