/******************************************************************************

  FileName    [ft-dom.c]

  PackageName [ft]

  Synopsis    [The domain manager for FT and IL models]

  Author      [Marco Pistore] 

  Copyright   [Copyright (C) 2003 by University of Trento.

  T-Tool is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  T-Tool is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information on the T-Tool see <http://dit.unitn.it/~ft>
  or email to <ft@dit.unitn.it>. Please report bugs to <ft@dit.unitn.it>.]

******************************************************************************/

#include <malloc.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include "ft.h"

#include "ft-int.h"

/* name set */

tNameSet mkNoNames()
{
  return NULL;
}

tNameSet mkConsNameSet(tName n, tNameSet ns)
{
  if (inNameSet(n, ns)) {
    return ns;
  }
  else {
    tNameSet res = ALLOC(struct _tNameSet);  
    res->name = n;
    res->next = ns;
    return res;
  }
}

tNameSet mkCupNameSet(tNameSet ns1, tNameSet ns2)
{
  tNameSet res = ns2;
  while (ns1) {
    res = mkConsNameSet(ns1->name, res);
    ns1 = ns1->next;
  }
  return res;
}

int inNameSet(tName n, tNameSet ns)
{
  while (ns) {
    if (isequalName(n,ns->name)) return 1;
    ns = ns->next;
  }
  return 0;
}

int isemptyNameSet(tNameSet names) 
{
  return (names == NULL);
}

tNameSet nextNameSet(tNameSet names)
{
  assert(names != NULL);
  return (names->next);
}

tName firstName(tNameSet names)
{
  assert(names != NULL);
  return (names->name);
}


/* name */

tName mkName(char * s)
{
  tName name = ALLOC(struct _tName);
  name->id = malloc(strlen(s)+1);
  strcpy(name->id, s);
  return name;
}

char * getNameString(tName n)
{
  return n->id;
}

int isequalName(tName n1, tName n2)
{
  return (strcmp(getNameString(n1), getNameString(n2)) == 0);
}

/* domain */

tDom mkEmptyDom()
{
  tDom dom = ALLOC(struct _tDom);
  dom->elements = NULL;
  dom->globals = NULL;
  return dom;
}

tDom addElement(tDom dom, tElement element)
{
  tElementSet elementSet = ALLOC(struct _tElementSet);
  elementSet->element = element;
  elementSet->next = dom->elements;
  dom->elements = elementSet;
  return dom;
}

tDom addGlobalProp(tDom dom, tProp prop)
{
  tPropSet globalSet = ALLOC(struct _tPropSet);
  globalSet->prop = prop;
  globalSet->next = dom->globals;
  dom->globals = globalSet;
  return dom;
}

tElementSet getDomElements(tDom dom)
{
  return dom->elements;
}

tElementSet getDomElsOfKind(tDom dom, tElKind k)
{
  tElementSet res = NULL;
  tElementSet* tail = &res;

  tElementSet tmp = dom->elements;
  while (tmp) {
    if (tmp->element->kind == k) {
      (*tail) = ALLOC(struct _tElementSet);
      (*tail)->element = tmp->element;
      (*tail)->next = NULL;
      tail = &((*tail)->next);
    }
    tmp = tmp->next;
  }
  return res;
}

tElementSet getDomEntities(tDom dom)
{
  return getDomElsOfKind(dom, entityEl);
}

tElementSet getDomActors(tDom dom)
{
  return getDomElsOfKind(dom, actorEl);
}

tElementSet getDomGoals(tDom dom)
{
  return getDomElsOfKind(dom, goalEl);
}

tElementSet getDomDeps(tDom dom)
{
  return getDomElsOfKind(dom, depEl);
}

tPropSet getDomGlobalProps(tDom dom)
{
  return dom->globals;
}

tElement getDomElementByName(tDom dom,tName name)
{
  tElementSet els = dom->elements;
  while (els) {
    tElement el = firstElement(els);

    if(isequalName(name, getElementName(el))) {
      return el;
    }

    els = nextElementSet(els);
  }
  
  return NULL;
}

/* element set */

int isemptyElementSet(tElementSet entities) 
{
  return (entities == NULL);
}

tElementSet nextElementSet(tElementSet entities)
{
  assert(entities != NULL);
  return (entities->next);
}

tElement firstElement(tElementSet entities)
{
  assert(entities != NULL);
  return (entities->element);
}

/* element */

tElKind getElementKind(tElement element)
{
  assert(element != NULL);
  return element->kind;
}

tName getElementName(tElement element)
{
  assert(element != NULL);
  return element->name;
}

tNameSet getElementRange(tElement element
) 
{
  assert(element != NULL);
  return element->range;
}     

tType getElementType(tElement element)
{
  assert(element != NULL);
  assert(element->kind == goalEl || element->kind == depEl);
  return element->type;
}

tMode getElementMode(tElement element)
{
  assert(element != NULL);
  assert(element->kind == goalEl || element->kind == depEl);
  return element->mode;
}

tElement getElementActor(tElement element)
{
  assert(element != NULL);
  assert(element->kind == goalEl);
  return element->actor;
}

tName getElementActorName(tElement element)
{
  assert(element != NULL);
  assert(element->kind == goalEl);
  return element->actorName;
}

tElement getElementDepender(tElement element)
{
  assert(element != NULL);
  assert(element->kind == depEl);
  return element->depender;
}

tName getElementDependerName(tElement element)
{
  assert(element != NULL);
  assert(element->kind == depEl);
  return element->dependerName;
}

tElement getElementDependee(tElement element)
{
  assert(element != NULL);
  assert(element->kind == depEl);
  return element->dependee;
}

tName getElementDependeeName(tElement element)
{
  assert(element != NULL);
  assert(element->kind == depEl);
  return element->dependeeName;
}

tAttrSet getElementAttrs(tElement element)
{
  assert(element != NULL);
  return element->attrSet;
}

tPropSet getElementCreates(tElement element)
{
  assert(element != NULL);
  return element->createSet;
}

tPropSet getElementInvars(tElement element)
{
  assert(element != NULL);
  return element->invarSet;
}

tPropSet getElementFulfills(tElement element)
{
  assert(element != NULL);
  return element->fulfillSet;
}

void setElementRange(tElement element, tNameSet range) 
{
  assert(element != NULL);
  element->range = range;
}

void setElementAttrs(tElement element, tAttrSet attrs) 
{
  assert(element != NULL);
  element->attrSet = attrs;
}

void setElementActor(tElement element, tElement actor)
{
  assert(element != NULL);
  assert(element->kind == goalEl);
  assert(element->actor == NULL);
  element->actor = actor;
}

void setElementDepender(tElement element, tElement depender)
{
  assert(element != NULL);
  assert(element->kind == depEl);
  assert(element->depender == NULL);
  element->depender = depender;
}

void setElementDependee(tElement element, tElement dependee)
{
  assert(element != NULL);
  assert(element->kind == depEl);
  assert(element->dependee == NULL);
  element->dependee = dependee;
}

/* class */

tElement mkClassEl(tName name, tAttrSet attrSet)
{
  return mkEntityEl(name, attrSet, NULL, NULL);
}

/* entity */

tElement mkEntityEl(tName name, 
		    tAttrSet attrSet, 
		    tPropSet createSet,
		    tPropSet invarSet)
{
  tElement element = ALLOC(struct _tElement);
  element->kind = entityEl;
  element->name = name;
  element->range = NULL;
  element->type = noType;
  element->mode = noMode;
  element->actor = NULL;
  element->depender = NULL;
  element->dependee = NULL;
  element->attrSet = attrSet;
  element->createSet = createSet;
  element->invarSet = invarSet;
  element->fulfillSet = NULL;
  return element;
}

/* actor */

tElement mkActorEl(tName name, 
		   tAttrSet attrSet, 
		   tPropSet createSet,
		   tPropSet invarSet)
{
  tElement element = ALLOC(struct _tElement);
  element->kind = actorEl;
  element->name = name;
  element->range = NULL;
  element->type = noType;
  element->mode = noMode;
  element->actor = NULL;
  element->depender = NULL;
  element->dependee = NULL;
  element->attrSet = attrSet;
  element->createSet = createSet;
  element->invarSet = invarSet;
  element->fulfillSet = NULL;
  return element;
}

/* goal */

tElement mkGoalEl(tName name, 
		  tType type,
		  tMode mode,
		  tName actorName,
		  tAttrSet attrSet, 
		  tPropSet createSet,
		  tPropSet invarSet,
		  tPropSet fulfillSet)
{
  tElement element = ALLOC(struct _tElement);
  element->kind = goalEl;
  element->name = name;
  element->range = NULL;
  element->type = type;
  element->mode = mode;
  element->actor = NULL;
  element->actorName = actorName;
  element->depender = NULL;
  element->dependerName = NULL;
  element->dependee = NULL;
  element->dependeeName = NULL;
  element->attrSet = attrSet;
  element->createSet = createSet;
  element->invarSet = invarSet;
  element->fulfillSet = fulfillSet;
  return element;
}

/* dependency */

tElement mkDepEl(tName name, 
		 tType type,
		 tMode mode,
		 tName dependerName,
		 tName dependeeName,
		 tAttrSet attrSet, 
		 tPropSet createSet,
		 tPropSet invarSet,
		 tPropSet fulfillSet)
{
  tElement element = ALLOC(struct _tElement);
  element->kind = depEl;
  element->name = name;
  element->range = NULL;
  element->type = type;
  element->mode = mode;
  element->actor = NULL;
  element->actorName = NULL;
  element->depender = NULL;
  element->dependerName = dependerName;
  element->dependee = NULL;
  element->dependeeName = dependeeName;
  element->attrSet = attrSet;
  element->createSet = createSet;
  element->invarSet = invarSet;
  element->fulfillSet = fulfillSet;
  return element;
}

/* property set */

tPropSet mkNoProps() 
{
  return NULL;
}

tPropSet addProp(tProp prop, tPropSet propSet)
{
  tPropSet propSetRes = ALLOC(struct _tPropSet);
  propSetRes->prop = prop;
  propSetRes->next = propSet;
  return propSetRes;
}

tPropSet addProps(tPropSet propSet1, tPropSet propSet2)
{
  if (propSet1 == NULL) return propSet2;
  return addProp(firstProp(propSet1), 
		 addProps(nextPropSet(propSet1), propSet2));
}

int isemptyPropSet(tPropSet props) 
{
  return (props == NULL);
}

tPropSet nextPropSet(tPropSet props)
{
  assert(props != NULL);
  return (props->next);
}

tProp firstProp(tPropSet props)
{
  assert(props != NULL);
  return (props->prop);
}

/* property */

tProp mkProp(tCategory category, tEvent event, tOrigin origin, tExpr expr)
{
  tProp prop = ALLOC(struct _tProp);
  prop->category = category;
  prop->event = event;
  prop->origin = origin;
  prop->expr = expr;
  return prop;
}

tCategory getPropCategory(tProp prop)
{
  return prop->category;
}

tEvent getPropEvent(tProp prop)
{ 
  return prop->event;
}

tOrigin getPropOrigin(tProp prop)
{
  return prop->origin;
}

tExpr getPropExpr(tProp prop) 
{
  return prop->expr;
}

/* attribute set */

tAttrSet mkNoAttributes()
{
  return NULL;
}

tAttrSet addAttribute(tAttr attr, tAttrSet attrSet)
{
  tAttrSet attrSetRes = ALLOC(struct _tAttrSet);
  attrSetRes->attr = attr;
  attrSetRes->next = attrSet;
  return attrSetRes;
}

int isemptyAttrSet(tAttrSet attrs) 
{
  return (attrs == NULL);
}

tAttrSet nextAttrSet(tAttrSet attrs)
{
  assert(attrs != NULL);
  return (attrs->next);
}

tAttr firstAttr(tAttrSet attrs)
{
  assert(attrs != NULL);
  return (attrs->attr);
}

/* attribute */

tAttr mkAttribute(tName name, tSort sort, tAttrFacet attrFacet)
{
  tAttr attr = ALLOC(struct _tAttr);
  attr->name = name;
  attr->sort = sort;
  attr->attrFacet = attrFacet;
  return attr;
}

tName getAttrName(tAttr attr)
{
  return attr->name;
}

tSort getAttrSort(tAttr attr)
{
  return attr->sort;
}

tAttrFacet getAttrFacet(tAttr attr)
{
  return attr->attrFacet;
}

/* facet */

tAttrFacet mkNoFacet()
{
  tAttrFacet attrFacet = ALLOC(struct _tAttrFacet);
  attrFacet->constantFacet = 0;
  return attrFacet;
}

tAttrFacet mkConstFacet()
{
  tAttrFacet attrFacet = ALLOC(struct _tAttrFacet);
  attrFacet->constantFacet = 1;
  return attrFacet;
}

int isConstFacet(tAttrFacet facet)
{
  return facet->constantFacet;
}

/* sort */

tSort mkBooleanSort()
{
  tSort sort = ALLOC(struct _tSort);
  sort->kind = boolSort;
  sort->className = NULL;
  sort->class = NULL;
  return sort;
}

tSort mkIntegerSort()
{
  tSort sort = ALLOC(struct _tSort);
  sort->kind = intSort;
  sort->className = NULL;
  sort->class = NULL;
  return sort;
}

tSort mkClassNameSort(tName name)
{
  tSort sort = ALLOC(struct _tSort);
  sort->kind = classSort;
  sort->className = name;
  sort->class = NULL; /* still to be assigned */
  return sort;
}

tSort mkClassSort(tElement el)
{
  tSort sort = ALLOC(struct _tSort);
  sort->kind = classSort;
  sort->className = getElementName(el);
  sort->class = el;
  return sort;
}

tSortKind getSortKind(tSort sort)
{
  return sort->kind;
}

tName getSortClassName(tSort sort)
{
  assert(sort->kind == classSort);
  return sort->className;
}

tElement getSortClass(tSort sort)
{
  assert(sort->kind == classSort);
  assert(sort->class);
  return sort->class;
}

void setSortClass(tSort sort, tElement class)
{
  assert(sort->kind == classSort);
  assert(sort->class == NULL);
  sort->class = class;
}

int isequalSort(tSort s1, tSort s2)
{
  if (s1->kind != s2->kind) return 0;
  if (s1->kind == classSort) {
    assert(s1->class && s2->class);
    if(s1->class != s2->class) return 0;
  }
  return 1;
}

/* mode */

tMode mkNoMode()
{
  return noMode;
}

tMode mkAchieveMode()
{
  return achieveMode;
}

tMode mkMaintainMode()
{
  return maintainMode;
}

tMode mkAchieveMaintainMode()
{
  return achieveMaintainMode;
}

tMode mkAvoidMode()
{
  return avoidMode;
}

/* type */

tType mkGoalType()
{
  return goalType;
}

tType mkSoftGoalType()
{
  return softGoalType;
}

tType mkTaskType()
{
  return taskType;
}

tType mkResourceType()
{
  return resourceType;
}

/* origin */

tOrigin mkNoOrigin()
{
  return noOrigin;
}

tOrigin mkDependerOrigin()
{
  return dependerOrigin;
}

tOrigin mkDependeeOrigin()
{
  return dependeeOrigin;
}

tOrigin mkDomainOrigin()
{
  return domainOrigin;
}

/* event */

tEvent mkNoEvent()
{
  return noEvent;
}

tEvent mkTriggerEvent()
{
  return triggerEvent;
}

tEvent mkConditionEvent()
{
  return conditionEvent;
}

tEvent mkDefinitionEvent()
{
  return definitionEvent;
}

/* category */

tCategory mkConstraintCategory()
{
  return constraintCategory;
}

tCategory mkAssertionCategory()
{
  return assertionCategory;
}

tCategory mkPossibilityCategory()
{
  return possibilityCategory;
}

/* atom */

tAtom mkNameAtom(tName name)
{
  tAtom atom = ALLOC(struct _tAtom);
  atom->kind = nameAtom;
  atom->name = name;
  return atom;
}

tAtom mkSelfAtom()
{
  tAtom atom = ALLOC(struct _tAtom);
  atom->kind = selfAtom;
  return atom;
}

tAtom mkActorAtom()
{
  tAtom atom = ALLOC(struct _tAtom);
  atom->kind = actorAtom;
  return atom;
}

tAtom mkDependeeAtom()
{
  tAtom atom = ALLOC(struct _tAtom);
  atom->kind = dependeeAtom;
  return atom;
}

tAtom mkDependerAtom()
{
  tAtom atom = ALLOC(struct _tAtom);
  atom->kind = dependerAtom;
  return atom;
}

tAtomKind getAtomKind(tAtom atom)
{
  return atom->kind;
}

tName getAtomName(tAtom atom)
{
  assert(atom->kind == nameAtom);
  return atom->name;
}

/* expression */

tExpr mkVar(tAtom atom)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = varExpr;
  expr->atom = atom;
  return expr;
}

tExpr mkDotVar(tExpr pref, tAtom atom)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = dotVarExpr;
  expr->atom = atom;
  expr->e1 = pref;
  return expr;
}

tExpr mkAnd(tExpr e1, tExpr e2)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = andExpr;
  expr->e1 = e1;
  expr->e2 = e2;
  return expr;
}

tExpr mkOr(tExpr e1, tExpr e2)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = orExpr;
  expr->e1 = e1;
  expr->e2 = e2;
  return expr;
}

tExpr mkImplies(tExpr e1, tExpr e2)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = impliesExpr;
  expr->e1 = e1;
  expr->e2 = e2;
  return expr;
}

tExpr mkIff(tExpr e1, tExpr e2)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = iffExpr;
  expr->e1 = e1;
  expr->e2 = e2;
  return expr;
}

tExpr mkNot(tExpr e)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = notExpr;
  expr->e1 = e;
  return expr;
}

tExpr mkNext(tExpr e)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = nextExpr;
  expr->e1 = e;
  return expr;
}

tExpr mkPrev(tExpr e)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = prevExpr;
  expr->e1 = e;
  return expr;
}

tExpr mkUntil(tExpr e1, tExpr e2)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = untilExpr;
  expr->e1 = e1;
  expr->e2 = e2;
  return expr;
}

tExpr mkSince(tExpr e1, tExpr e2)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = sinceExpr;
  expr->e1 = e1;
  expr->e2 = e2;
  return expr;
}

tExpr mkFinally(tExpr e)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = finallyExpr;
  expr->e1 = e;
  return expr;
}

tExpr mkPast(tExpr e)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = pastExpr;
  expr->e1 = e;
  return expr;
}

tExpr mkGlobally(tExpr e)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = globallyExpr;
  expr->e1 = e;
  return expr;
}

tExpr mkPastGlobally(tExpr e)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = pastGloballyExpr;
  expr->e1 = e;
  return expr;
}

tExpr mkForall(tName name, tName sort, tExpr e)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = forallExpr;
  expr->name = name;
  expr->sort = sort;
  expr->e1 = e;
  return expr;
}

tExpr mkExists(tName name, tName sort, tExpr e)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = existsExpr;
  expr->name = name;
  expr->sort = sort;
  expr->e1 = e;
  return expr;
}

tExpr mkNumber(int num)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = numberExpr;
  expr->num = num;
  return expr;
}

tExpr mkFalse()
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = falseExpr;
  return expr;
}

tExpr mkTrue()
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = trueExpr;
  return expr;
}

tExpr mkEqual(tExpr e1, tExpr e2)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = equalExpr;
  expr->e1 = e1;
  expr->e2 = e2;
  return expr;
}

tExpr mkNotEqual(tExpr e1, tExpr e2)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = notEqualExpr;
  expr->e1 = e1;
  expr->e2 = e2;
  return expr;
}

tExpr mkFulfilled(tExpr e)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = fulfilledExpr;
  expr->e1 = e;
  return expr;
}

tExpr mkJustFulfilled(tExpr e)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = justFulfilledExpr;
  expr->e1 = e;
  return expr;
}

tExpr mkJustCreated(tExpr e)
{
  tExpr expr = ALLOC(struct _tExpr);
  expr->kind = justCreatedExpr;
  expr->e1 = e;
  return expr;
}

tSort getExprType(tExpr e)
{
  assert(e->type);
  return e->type;
}

tExprKind getExprKind(tExpr e)
{
  return e->kind;
}

tAtom getExprVar(tExpr e)
{
  assert(e->kind == varExpr);
  return e->atom;
}

tExpr getExprDotPrefix(tExpr e)
{
  assert(e->kind == dotVarExpr);
  return e->e1;  
}

tAtom getExprDotSuffix(tExpr e)
{
  assert(e->kind == dotVarExpr);
  return e->atom;  
}

tExpr getExprFirstOp(tExpr e)
{
  assert(e->kind == andExpr || e->kind == orExpr || 
	 e->kind == impliesExpr || e->kind == iffExpr ||  
	 e->kind == untilExpr || e->kind == sinceExpr || 
	 e->kind == equalExpr || e->kind == notEqualExpr);
  return e->e1;  
}

tExpr getExprSecondOp(tExpr e)
{
  assert(e->kind == andExpr || e->kind == orExpr || 
	 e->kind == impliesExpr || e->kind == iffExpr ||  
	 e->kind == untilExpr || e->kind == sinceExpr || 
	 e->kind == equalExpr || e->kind == notEqualExpr);
  return e->e2;  
}

tExpr getExprOp(tExpr e)
{
  assert(e->kind == notExpr || e->kind == nextExpr || 
	 e->kind == prevExpr || e->kind == finallyExpr ||  
	 e->kind == pastExpr || e->kind == globallyExpr || 
	 e->kind == pastGloballyExpr || e->kind == fulfilledExpr ||
	 e->kind == justFulfilledExpr || e->kind == justCreatedExpr);
  return e->e1;  
}

tName getExprQuantVar(tExpr e)
{
  assert(e->kind == forallExpr || e->kind == existsExpr);
  return e->name;
}

tName getExprQuantClass(tExpr e)
{
  assert(e->kind == forallExpr || e->kind == existsExpr);
  return e->sort;
}

tSort getExprQuantType(tExpr e)
{
  assert(e->kind == forallExpr || e->kind == existsExpr);
  return e->sortClass;
}

tExpr getExprQuantBody(tExpr e)
{
  assert(e->kind == forallExpr || e->kind == existsExpr);
  return e->e1;
}

int getExprNumber(tExpr e)
{
  assert(e->kind == numberExpr);
  return e->num;
}

void setExprType(tExpr e, tSort t)
{
  /* assert(! (e->type));*/
  e->type = t;
}

void setExprQuantType(tExpr e, tSort t)
{
  assert(e->kind == forallExpr || e->kind == existsExpr);
  assert(! (e->sortClass));
  e->sortClass = t;
}

tNameSet getExprFvAux(tExpr e, tNameSet bv)
{
  switch(getExprKind(e)) {
  case varExpr:
    if (getAtomKind(getExprVar(e)) == dependerAtom) {
      return mkConsNameSet(mkName("depender"),NULL);
    } 
    else if (getAtomKind(getExprVar(e)) == dependeeAtom) {
      return mkConsNameSet(mkName("dependee"),NULL);
    } 
    else if (getAtomKind(getExprVar(e)) == actorAtom) {
      return mkConsNameSet(mkName("actor"),NULL);
    } 
    else if (getAtomKind(getExprVar(e)) == nameAtom && 
	     !inNameSet(getAtomName(getExprVar(e)),bv)) {
      return mkConsNameSet(getAtomName(getExprVar(e)),NULL);
    }
    else {
      return NULL;
    }
  case dotVarExpr:
    return getExprFvAux(getExprDotPrefix(e),bv);
  case andExpr:
  case orExpr:
  case impliesExpr:
  case iffExpr:
  case untilExpr:
  case sinceExpr:
  case equalExpr:
  case notEqualExpr:
    return mkCupNameSet(getExprFvAux(getExprFirstOp(e),bv), 
			getExprFvAux(getExprSecondOp(e), bv));
  case notExpr:
  case nextExpr:
  case prevExpr:
  case finallyExpr:
  case pastExpr:
  case globallyExpr:
  case pastGloballyExpr:
  case fulfilledExpr:
  case justFulfilledExpr:
  case justCreatedExpr:
    return getExprFvAux(getExprOp(e),bv);
  case numberExpr:
  case falseExpr:
  case trueExpr:
    return NULL;
  case forallExpr:
  case existsExpr:
    return getExprFvAux(getExprQuantBody(e), mkConsNameSet(getExprQuantVar(e),bv));
  default:
    abort();
  }
}

tNameSet getExprFv(tExpr expr)
{
  return getExprFvAux(expr, mkNoNames());
}
