%{
/******************************************************************************

  FileName    [ft-gra.c]

  PackageName [ft]

  Synopsis    [The parser of the FT language]

  Author      [Marco Pistore] 

  Copyright   [Copyright (C) 2003 by University of Trento.

  T-Tool is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  T-Tool is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information on the T-Tool see <http://dit.unitn.it/~ft>
  or email to <ft@dit.unitn.it>. Please report bugs to <ft@dit.unitn.it>.]

******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include "ft.h"

#define YYERROR_VERBOSE

int ft_error(char *s);
int ft_lex();

static tDom dom;
tName curr_actor;

%}

%name-prefix="ft_"

%union{
  int num;
  tName name;
  tElement element;
  tMode mode;
  tAttr attr;
  tAttrFacet attrFacet;
  tAttrSet attrSet;
  tSort sort;
  tType type;
  tProp prop;
  tPropSet propSet;
  tCategory category;
  tEvent event;
  tOrigin origin;
  tExpr expr;
  tAtom atom;
}

%token<name> NAME
%token<num> NUMBER
%token ENTITY ACTOR GOAL DEPENDENCY ATTRIBUTE CONSTANT INTEGER BOOLEAN
%token DEPENDER DEPENDEE TYPE SOFTGOAL TASK RESOURCE MODE ACHIEVE
%token MAINTAIN ACHIEVE_AND_MAINTAIN AVOID GLOBAL CREATION INVAR FULFILLMENT
%token CONSTRAINT ASSERTION POSSIBILITY TRIGGER CONDITION DEFINITION FOR 
%token DOMAIN TRUE FALSE FORALL EXISTS SELF FULFILLED JUSTFULFILLED JUSTCREATED

%right IMPLIES
%left  IFF
%left  OR
%left  AND
%left  NOT
%left  UNTIL SINCE
%left  NEXT PREV GLOBALLY PAST_GLOBALLY FINALLY PAST
%left  EQUAL NOTEQUAL

%type<element> entity actor goal dependency
%type<mode> mode_opt mode
%type<type> type
%type<attrSet> attributes_opt attribute_list
%type<attr> attribute
%type<attrFacet> attribute_facets constant_facet_opt
%type<sort> sort basic_sort
%type<propSet> creation_properties_opt creation_property_list
%type<propSet> invar_properties_opt invar_property_list
%type<propSet> fulfill_properties_opt fulfill_property_list
%type<prop> global_property creation_property invar_property fulfill_property
%type<category> property_category
%type<event> event_category
%type<origin> property_origin
%type<expr> temporal_formula expr var
%type<atom> atom

%%

model	: element_list
	;

element_list
	: element
	| element element_list 
	;

element	: entity
          {
            dom = addElement(dom,$1);
          }
	| actor
          {
            dom = addElement(dom,$1);
          }
	| dependency
          {
            dom = addElement(dom,$1);
          }
	| goal
          {
            dom = addElement(dom,$1);
          }
	| global
	;

entity	: ENTITY NAME
	  attributes_opt
	  creation_properties_opt
	  invar_properties_opt
	  {
	    $$ = mkEntityEl($2,$3,$4,$5);
          }
	;

attributes_opt
	: /* void */
          { 
            $$ = mkNoAttributes();
          }
	| ATTRIBUTE attribute_list
          { 
            $$ = $2; 
          }
	;

attribute_list
	: attribute 
          { 
            $$ = addAttribute($1, mkNoAttributes()); 
          }
	| attribute attribute_list
          { 
            $$ = addAttribute($1, $2); 
          }
	;

attribute
	: attribute_facets NAME ':' sort
          {
	    $$ = mkAttribute($2,$4,$1);
          }
	;

attribute_facets
	: constant_facet_opt
          { 
            $$ = $1;
          }
	;

constant_facet_opt
	: /* void */
          {
            $$ = mkNoFacet();
          }
	| CONSTANT
          {
            $$ = mkConstFacet();
          }
	;

sort	: NAME
          {
            $$ = mkClassNameSort($1);
          }
	| basic_sort
          {
            $$ = $1;
          }          
	;

basic_sort
	: INTEGER
          {
            $$ = mkIntegerSort();
          }
	| BOOLEAN
          {
            $$ = mkBooleanSort();
          }
	;

actor	: ACTOR NAME 
          { 
            curr_actor = $2; 
          }
	  attributes_opt
	  creation_properties_opt
	  invar_properties_opt
          {
            $$ = mkActorEl($2,$4,$5,$6);
          }
	;

goal
	: type NAME
	  mode_opt
          ACTOR NAME
	  attributes_opt
	  creation_properties_opt
	  invar_properties_opt
	  fulfill_properties_opt
          {
            $$ = mkGoalEl($2,$1,$3,$5,$6,$7,$8,$9);
          }
        ;

dependency
	: DEPENDENCY NAME
	  TYPE type
	  mode_opt
	  DEPENDER NAME
	  DEPENDEE NAME
	  attributes_opt
	  creation_properties_opt
	  invar_properties_opt
	  fulfill_properties_opt
          {
            $$ = mkDepEl($2,$4,$5,$7,$9,$10,$11,$12,$13);
          }
	| type DEPENDENCY NAME
	  mode_opt
	  DEPENDER NAME
	  DEPENDEE NAME
	  attributes_opt
	  creation_properties_opt
	  invar_properties_opt
	  fulfill_properties_opt
          {
            $$ = mkDepEl($3,$1,$4,$6,$8,$9,$10,$11,$12);
          }
        ;

type	: GOAL
	  {
	    $$ = mkGoalType();
          }
	| SOFTGOAL
	  {
	    $$ = mkSoftGoalType();
          }
	| TASK
	  {
	    $$ = mkTaskType();
          }
	| RESOURCE
	  {
	    $$ = mkResourceType();
          }
	;

mode_opt: /* void */
          {
	    $$ = mkNoMode();
          }
        | mode
          {
	    $$ = $1;
	  }
        ;

mode	: MODE ACHIEVE
          {
	    $$ = mkAchieveMode();
          }         
	| MODE MAINTAIN
          {
	    $$ = mkMaintainMode();
          }         
	| MODE ACHIEVE_AND_MAINTAIN
          {
	    $$ = mkAchieveMaintainMode();
          }         
	| MODE AVOID
          {
	    $$ = mkAvoidMode();
          }         
	;

global
	: GLOBAL global_property_list
	;

creation_properties_opt
	: /* void */
          {
	    $$ = mkNoProps();
          }
	| CREATION creation_property_list creation_properties_opt
          {
	    $$ = addProps($2,$3);
          }
	;

invar_properties_opt
	: /* void */
          {
	    $$ = mkNoProps();
          }
	| INVAR invar_property_list invar_properties_opt
          {
	    $$ = addProps($2,$3);
          }
	;

fulfill_properties_opt
	: /* void */
          {
	    $$ = mkNoProps();
          }
	| FULFILLMENT fulfill_property_list fulfill_properties_opt
          {
	    $$ = addProps($2,$3);
          }
	;

global_property_list
	: global_property
	  {
	    dom = addGlobalProp(dom,$1);
	  }
	| global_property 
	  {
	    dom = addGlobalProp(dom,$1);
	  }
          global_property_list
	;

global_property
	: property_category temporal_formula
          {
	    $$ = mkProp($1,mkNoEvent(),mkNoOrigin(),$2);
          }
	;

creation_property_list
	: creation_property
	  {
            $$ = addProp($1,mkNoProps());
          }
	| creation_property creation_property_list
	  {
            $$ = addProp($1,$2);
          }
	;

creation_property
	: property_category event_category property_origin temporal_formula
          {
	    $$ = mkProp($1,$2,$3,$4);
          }
	;

invar_property_list
	: invar_property
	  {
            $$ = addProp($1,mkNoProps());
          }
	| invar_property invar_property_list
	  {
            $$ = addProp($1,$2);
          }
	;

invar_property
	: property_category property_origin temporal_formula
          {
	    $$ = mkProp($1,mkNoEvent(),$2,$3);
          }
	;

fulfill_property_list
	: fulfill_property
	  {
            $$ = addProp($1,mkNoProps());
          }
	| fulfill_property fulfill_property_list
	  {
            $$ = addProp($1,$2);
          }
	;

fulfill_property
	: property_category event_category property_origin temporal_formula
          {
	    $$ = mkProp($1,$2,$3,$4);
          }
	;

property_category
	: /* void */
	  {
	    $$ = mkConstraintCategory();
          }
	| CONSTRAINT
	  {
	    $$ = mkConstraintCategory();
          }
	| ASSERTION
	  {
	    $$ = mkAssertionCategory();
          }
	| POSSIBILITY
	  {
	    $$ = mkPossibilityCategory();
          }
	;

event_category
	: TRIGGER
	  {
	    $$ = mkTriggerEvent();
          }
	| CONDITION
	  {
	    $$ = mkConditionEvent();
          }
	| DEFINITION
	  {
	    $$ = mkDefinitionEvent();
          }
	;

property_origin
	: /* void */
	  {
	    $$ = mkNoOrigin();
          }
	| FOR DEPENDER
	  {
	    $$ = mkDependerOrigin();
          }
	| FOR DEPENDEE
	  {
	    $$ = mkDependeeOrigin();
          }
	| FOR DOMAIN
	  {
	    $$ = mkDomainOrigin();
          }
	;

temporal_formula
	: expr
          {
	    $$ = $1;
          }
        ;

atom    : DEPENDER
          {
	    $$ = mkDependerAtom();
	  }
	| DEPENDEE
          {
	    $$ = mkDependeeAtom();
	  }
	| ACTOR
          {
	    $$ = mkActorAtom();
	  }
	| NAME
          {
	    $$ = mkNameAtom($1);
	  }
        ;

var     : SELF
          {
	    $$ = mkVar(mkSelfAtom());
	  }
	| atom
          {
	    $$ = mkVar($1);
          }
        | var '.' atom
          {
	    $$ = mkDotVar($1,$3);
	  }
        ;

expr    : var
          {
            $$ = $1;
	  }
	| expr AND expr
          {
	    $$ = mkAnd($1,$3);
          }
	| expr OR expr
          {
	    $$ = mkOr($1,$3);
          }
	| expr IMPLIES expr
          {
	    $$ = mkImplies($1,$3);
          }
	| expr IFF expr
          {
	    $$ = mkIff($1,$3);
          }
	| NOT expr
          {
	    $$ = mkNot($2);
          }
	| NEXT expr
          {
	    $$ = mkNext($2);
          }
	| PREV expr
          {
	    $$ = mkPrev($2);
          }
	| expr UNTIL expr
	  {
            $$ = mkUntil($1,$3);
	  }
	| expr SINCE expr
	  {
            $$ = mkSince($1,$3);
	  }
	| GLOBALLY expr
          {
	    $$ = mkGlobally($2);
          }
	| PAST_GLOBALLY expr
          {
	    $$ = mkPastGlobally($2);
          }
	| FINALLY expr
          {
	    $$ = mkFinally($2);
          }
	| PAST expr
          {
	    $$ = mkPast($2);
          }
	| '(' expr ')'
          {
            $$ = $2;
          }
	| FORALL NAME ':' NAME '(' expr ')'
	  {
            $$ = mkForall($2,$4,$6);
	  }
	| EXISTS NAME ':' NAME '(' expr ')'
	  {
            $$ = mkExists($2,$4,$6);
	  }
	| NUMBER
	  {
            $$ = mkNumber($1);
	  }
	| FALSE
	  {
            $$ = mkFalse();
	  }
        | TRUE
	  {
            $$ = mkTrue();
	  }
	| expr EQUAL expr
	  {
            $$ = mkEqual($1,$3);
	  }
	| expr NOTEQUAL expr
	  {
            $$ = mkNotEqual($1,$3);
	  }
	| FULFILLED '(' var ')'
	  {
            $$ = mkFulfilled($3);
	  }
	| JUSTFULFILLED '(' var ')'
	  {
            $$ = mkJustFulfilled($3);
	  }
	| JUSTCREATED '(' var ')'
	  {
            $$ = mkJustCreated($3);
	  }
	;

%%

int ft_error(char *s)
{
  extern int ft_lineno;
  fprintf(stderr, "Error: %s at line %d\n", s, ft_lineno);
  abort();
}

tDom parseFT(FILE * f)
{
  extern FILE * ft_in;

  tDom parseDom;
  
  /* Using global dom variable for parsing */
  dom = mkEmptyDom();

  ft_in = f;
  ft_parse();
  
  parseDom = dom;
  dom = NULL;

  return parseDom;
}
