/******************************************************************************

  FileName    [ft-print.c]

  PackageName [ft]

  Synopsis    [The print routines of the ft package]

  Author      [Marco Pistore] 

  Copyright   [Copyright (C) 2003 by University of Trento.

  T-Tool is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  T-Tool is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information on the T-Tool see <http://dit.unitn.it/~ft>
  or email to <ft@dit.unitn.it>. Please report bugs to <ft@dit.unitn.it>.]

******************************************************************************/

#include <malloc.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "ft.h"

char * getTypeString(tType type)
{
  switch(type) {
  case goalType:
    return "GOAL";
  case softGoalType:
    return "SOFTGOAL";
  case taskType:
    return "TASK";
  case resourceType:
    return "RESOURCE";
  default:
    abort();
  }
};

char * getModeString(tMode mode)
{
  switch(mode) {
  case achieveMode:
    return "Achieve";
  case maintainMode:
    return "Maintain";
  case achieveMaintainMode:
    return "AchieveMaintain";
  case avoidMode:
    return "Avoid";
  default:
    abort();
  }
}

char * getSortString(tSort sort)
{
  switch (getSortKind(sort)) {
  case intSort:
    return "integer";
  case boolSort:
    return "boolean";
  case classSort:
    return getNameString(getSortClassName(sort));
  default:
    abort();
  }
}

char * getCategoryString(tCategory cat)
{
  switch (cat) {
  case constraintCategory:
    return "CONSTRAINT";
  case assertionCategory:
    return "ASSERTION";
  case possibilityCategory:
    return "POSSIBILITY";
  default:
    abort();
  }
}

char * getEventString(tEvent event)
{
  switch (event) {
  case triggerEvent:
    return "TRIGGER";
  case conditionEvent:
    return "CONDITION";
  case definitionEvent:
    return "DEFINITION";
  default:
    abort();
  }
}

char * getOriginString(tOrigin origin)
{
  switch (origin) {
  case dependerOrigin:
    return "FOR DEPENDER";
  case dependeeOrigin:
    return "FOR DEPENDEE";
  case domainOrigin:
    return "FOR DOMAIN";
  default:
    abort();
  }
}

void printAttrs(FILE * f, tAttrSet attrs)
{
  if (isemptyAttrSet(attrs)) return;
  fprintf(f, "  ATTRIBUTE\n");
  while(! isemptyAttrSet(attrs)) {
    tAttr attr = firstAttr(attrs);
    fprintf(f, "    %s : ", getNameString(getAttrName(attr)));
    if (isConstFacet(getAttrFacet(attr))) {
      fprintf(f, "CONSTANT ");
    }
    fprintf(f, "%s\n", getSortString(getAttrSort(attr)));
    attrs = nextAttrSet(attrs);
  }
}

void printAtom(FILE * f, tAtom a)
{
  switch(getAtomKind(a)) {
  case nameAtom:
    fprintf(f,"%s", getNameString(getAtomName(a)));
    break;
  case selfAtom:
    fprintf(f,"self");
    break;
  case actorAtom:
    fprintf(f,"actor");
    break;
  case dependeeAtom:
    fprintf(f,"dependee");
    break;
  case dependerAtom:
    fprintf(f,"depender");
    break;
  default:
    abort();
  }
}

void printExpr(FILE * f, tExpr e)
{
  switch(getExprKind(e)) {
  case varExpr:
    printAtom(f,getExprVar(e));
    break;
  case dotVarExpr:
    printExpr(f,getExprDotPrefix(e));
    fprintf(f,".");
    printAtom(f,getExprDotSuffix(e));
    break;
  case andExpr:
    fprintf(f,"( ");
    printExpr(f,getExprFirstOp(e));
    fprintf(f," & ");
    printExpr(f,getExprSecondOp(e));
    fprintf(f," )");
    break;
  case orExpr:
    fprintf(f,"( ");
    printExpr(f,getExprFirstOp(e));
    fprintf(f," | ");
    printExpr(f,getExprSecondOp(e));
    fprintf(f," )");
    break;
  case impliesExpr:
    fprintf(f,"( ");
    printExpr(f,getExprFirstOp(e));
    fprintf(f," -> ");
    printExpr(f,getExprSecondOp(e));
    fprintf(f," )");
    break;
  case iffExpr:
    fprintf(f,"( ");
    printExpr(f,getExprFirstOp(e));
    fprintf(f," <-> ");
    printExpr(f,getExprSecondOp(e));
    fprintf(f," )");
    break;
  case untilExpr:
    fprintf(f,"( ");
    printExpr(f,getExprFirstOp(e));
    fprintf(f," U ");
    printExpr(f,getExprSecondOp(e));
    fprintf(f," )");
    break;
  case sinceExpr:
    fprintf(f,"( ");
    printExpr(f,getExprFirstOp(e));
    fprintf(f," S ");
    printExpr(f,getExprSecondOp(e));
    fprintf(f," )");
    break;
  case equalExpr:
    fprintf(f,"( ");
    printExpr(f,getExprFirstOp(e));
    fprintf(f," = ");
    printExpr(f,getExprSecondOp(e));
    fprintf(f," )");
    break;
  case notEqualExpr:
    fprintf(f,"( ");
    printExpr(f,getExprFirstOp(e));
    fprintf(f," != ");
    printExpr(f,getExprSecondOp(e));
    fprintf(f," )");
    break;
  case notExpr:
    fprintf(f,"( ! ");
    printExpr(f,getExprOp(e));
    fprintf(f," )");
    break;
  case nextExpr:
    fprintf(f,"( X ");
    printExpr(f,getExprOp(e));
    fprintf(f," )");
    break;
  case prevExpr:
    fprintf(f,"( Y ");
    printExpr(f,getExprOp(e));
    fprintf(f," )");
    break;
  case finallyExpr:
    fprintf(f,"( F ");
    printExpr(f,getExprOp(e));
    fprintf(f," )");
    break;
  case pastExpr:
    fprintf(f,"( O ");
    printExpr(f,getExprOp(e));
    fprintf(f," )");
    break;
  case globallyExpr:
    fprintf(f,"( G ");
    printExpr(f,getExprOp(e));
    fprintf(f," )");
    break;
  case pastGloballyExpr:
    fprintf(f,"( H ");
    printExpr(f,getExprOp(e));
    fprintf(f," )");
    break;
  case numberExpr:
    fprintf(f,"%d", getExprNumber(e));
    break;
  case falseExpr:
    fprintf(f,"FALSE");
    break;
  case trueExpr:
    fprintf(f,"TRUE");
    break;
  case forallExpr:
    fprintf(f,"( FORALL %s: %s ( ", 
	   getNameString(getExprQuantVar(e)),
	   getNameString(getExprQuantClass(e)));
    printExpr(f,getExprQuantBody(e));
    fprintf(f," ) )");
    break;
  case existsExpr:
    fprintf(f,"( EXISTS %s: %s ( ", 
	   getNameString(getExprQuantVar(e)),
	   getNameString(getExprQuantClass(e)));
    printExpr(f,getExprQuantBody(e));
    fprintf(f," ) )");
    break;
  case fulfilledExpr:
    fprintf(f,"( Fulfilled ( ");
    printExpr(f,getExprOp(e));
    fprintf(f," ) )");
    break;
  case justFulfilledExpr:
    fprintf(f,"( JustFulfilled ( ");
    printExpr(f,getExprOp(e));
    fprintf(f," ) )");
    break;
  case justCreatedExpr:
    fprintf(f,"( JustCreated ( ");
    printExpr(f,getExprOp(e));
    fprintf(f," ) )");
    break;
  default:
    abort();
  }
}

void printProp(FILE * f, tProp prop)
{
  fprintf(f,"%s ", getCategoryString(getPropCategory(prop)));
  if (getPropEvent(prop) != noEvent) {
    fprintf(f,"%s ", getEventString(getPropEvent(prop)));
  }
  if (getPropOrigin(prop) != noOrigin) {
    fprintf(f,"%s ", getOriginString(getPropOrigin(prop)));
  }
  printExpr(f,getPropExpr(prop));
}

void printCreation(FILE * f, tPropSet creates)
{
  if (isemptyPropSet(creates)) return;
  fprintf(f,"  CREATION\n");

  while (!isemptyPropSet(creates)) {
    fprintf(f,"    ");
    printProp(f,firstProp(creates));
    fprintf(f,"\n");
    creates = nextPropSet(creates);
  }
}

void printInvariant(FILE * f, tPropSet invars)
{
  if (isemptyPropSet(invars)) return;
  fprintf(f,"  INVARIANT\n");

  while (!isemptyPropSet(invars)) {
    fprintf(f,"    ");
    printProp(f,firstProp(invars));
    fprintf(f,"\n");
    invars = nextPropSet(invars);
  }
}

void printFulfillment(FILE * f, tPropSet fulfills)
{
  if (isemptyPropSet(fulfills)) return;
  fprintf(f,"  FULFILLMENT\n");
  
  while(!isemptyPropSet(fulfills)) {
    fprintf(f,"    ");
    printProp(f,firstProp(fulfills));
    fprintf(f,"\n");
    fulfills = nextPropSet(fulfills);
  }
}

void printFT(FILE * f, tDom dom) 
{
  tElementSet entities;
  tElementSet actors;
  tElementSet goals;
  tElementSet deps;
  tPropSet globals;

  for (entities = getDomEntities(dom);
       !isemptyElementSet(entities); 
       entities = nextElementSet(entities)) {
    tElement entity = firstElement(entities);

    fprintf(f,"ENTITY %s\n", getNameString(getElementName(entity)));
    printAttrs(f,getElementAttrs(entity));
    printCreation(f,getElementCreates(entity));
    printInvariant(f,getElementInvars(entity));

    fprintf(f,"\n");
  }
    
  for (actors = getDomActors(dom);
       !isemptyElementSet(actors);
       actors = nextElementSet(actors)) {
    tElement actor = firstElement(actors);

    fprintf(f,"ACTOR %s\n", getNameString(getElementName(actor)));
    printAttrs(f,getElementAttrs(actor));
    printCreation(f,getElementCreates(actor));
    printInvariant(f,getElementInvars(actor));

    fprintf(f,"\n");
  }

  for (goals = getDomGoals(dom);
       !isemptyElementSet(goals);
       goals = nextElementSet(goals)) {
    tElement goal = firstElement(goals);

    fprintf(f,"%s %s\n", 
	   getTypeString(getElementType(goal)), 
	   getNameString(getElementName(goal)));
    fprintf(f,"  MODE %s\n", getModeString(getElementMode(goal)));
    fprintf(f,"  ACTOR %s\n", getNameString(getElementActorName(goal)));
    printAttrs(f,getElementAttrs(goal));
    printCreation(f,getElementCreates(goal));
    printInvariant(f,getElementInvars(goal));
    printFulfillment(f,getElementFulfills(goal));

    fprintf(f,"\n");
  }

  for (deps = getDomDeps(dom);
       !isemptyElementSet(deps);
       deps = nextElementSet(deps)) {
    tElement dep = firstElement(deps);

    fprintf(f,"DEPENDENCY %s\n", getNameString(getElementName(dep)));
    fprintf(f,"  TYPE %s\n", getTypeString(getElementType(dep)));
    fprintf(f,"  MODE %s\n", getModeString(getElementMode(dep)));
    fprintf(f,"  DEPENDER %s\n", getNameString(getElementDependerName(dep)));
    fprintf(f,"  DEPENDEE %s\n", getNameString(getElementDependeeName(dep)));
    printAttrs(f,getElementAttrs(dep));
    printCreation(f,getElementCreates(dep));
    printInvariant(f,getElementInvars(dep));
    printFulfillment(f,getElementFulfills(dep));

    fprintf(f,"\n");
  }

  for (globals = getDomGlobalProps(dom);
       !isemptyPropSet(globals);
       globals = nextPropSet(globals)) {
    fprintf(f,"GLOBAL\n  ");
    printProp(f,firstProp(globals));
    fprintf(f,"\n\n");
  }
}

void printIL(FILE * f, tDom dom) 
{
  tElementSet entities;
  tPropSet globals;

  for (entities = getDomEntities(dom);
       !isemptyElementSet(entities); 
       entities = nextElementSet(entities)) {
    tElement entity = firstElement(entities);
    tAttrSet attrs = getElementAttrs(entity);

    fprintf(f,"CLASS %s\n", getNameString(getElementName(entity)));
    while(! isemptyAttrSet(attrs)) {
      tAttr attr = firstAttr(attrs);
      fprintf(f,"  %s : %s\n", 
	     getNameString(getAttrName(attr)),
	     getSortString(getAttrSort(attr)));
      attrs = nextAttrSet(attrs);
    }
    fprintf(f,"\n");
  }

  for (globals = getDomGlobalProps(dom);
       !isemptyPropSet(globals);
       globals = nextPropSet(globals)) {
    printProp(f,firstProp(globals));
    fprintf(f,"\n\n");
  }
}

void printBIL(FILE * f, tDom dom) 
{
  tElementSet entities;
  tPropSet globals;

  for (entities = getDomEntities(dom);
       !isemptyElementSet(entities); 
       entities = nextElementSet(entities)) {
    tElement entity = firstElement(entities);
    tAttrSet attrs = getElementAttrs(entity);

    fprintf(f,"CLASS %s\n", getNameString(getElementName(entity)));
    while(! isemptyAttrSet(attrs)) {
      tAttr attr = firstAttr(attrs);
      if (getSortKind(getAttrSort(attr)) == classSort) {
	tNameSet range = getElementRange(getDomElementByName(dom,getSortClassName(getAttrSort(attr))));
	fprintf(f,"  VAR %s : { ", 
		getNameString(getAttrName(attr)));
	while(! isemptyNameSet(range)) {
	  tName n = firstName(range);
	  fprintf(f,"%s", getNameString(n));
	  range = nextNameSet(range);
	  if (! isemptyNameSet(range)) {
	    fprintf(f,", ");
	  }
	}
	fprintf(f," };\n");
      } else {
	fprintf(f,"  %s : %s\n", 
		getNameString(getAttrName(attr)),
		getSortString(getAttrSort(attr)));
      }
      attrs = nextAttrSet(attrs);
    }
    fprintf(f,"\n");
  }

  for (globals = getDomGlobalProps(dom);
       !isemptyPropSet(globals);
       globals = nextPropSet(globals)) {
    printProp(f,firstProp(globals));
    fprintf(f,"\n\n");
  }
}

int isSMVPropExpr(tExpr e) 
{
  switch(getExprKind(e)) {
  case varExpr:
  case dotVarExpr:
  case equalExpr:
  case notEqualExpr:
  case numberExpr:
  case falseExpr:
  case trueExpr:
    return 1;
  case untilExpr:
  case sinceExpr:
  case nextExpr:
  case prevExpr:
  case finallyExpr:
  case pastExpr:
  case globallyExpr:
  case pastGloballyExpr:
    return 0;
  case andExpr:
  case orExpr:
  case impliesExpr:
  case iffExpr:
    return isSMVPropExpr(getExprFirstOp(e)) && isSMVPropExpr(getExprSecondOp(e));
  case notExpr:
    return isSMVPropExpr(getExprOp(e));
  default:
    abort();
  }
}

int isSMVSimpleXExpr(tExpr e) 
{
  switch(getExprKind(e)) {
  case varExpr:
  case dotVarExpr:
  case equalExpr:
  case notEqualExpr:
  case numberExpr:
  case falseExpr:
  case trueExpr:
    return 1;
  case untilExpr:
  case sinceExpr:
  case prevExpr:
  case finallyExpr:
  case pastExpr:
  case globallyExpr:
  case pastGloballyExpr:
    return 0;
  case nextExpr:
    return isSMVPropExpr(getExprOp(e));
  case andExpr:
  case orExpr:
  case impliesExpr:
  case iffExpr:
    return isSMVSimpleXExpr(getExprFirstOp(e)) && isSMVSimpleXExpr(getExprSecondOp(e));
  case notExpr:
    return isSMVSimpleXExpr(getExprOp(e));
  default:
    abort();
  }
}

void printSMVSimpleXExpr(FILE * f, tExpr e)
{
  switch(getExprKind(e)) {
  case varExpr:
  case dotVarExpr:
  case equalExpr:
  case notEqualExpr:
  case numberExpr:
  case falseExpr:
  case trueExpr:
    printExpr(f,e);
    break;
  case nextExpr:
    fprintf(f,"( next( ");
    printExpr(f,getExprOp(e));
    fprintf(f," ) )");
    break;
  case andExpr:
    fprintf(f,"( ");
    printSMVSimpleXExpr(f,getExprFirstOp(e));
    fprintf(f," & ");
    printSMVSimpleXExpr(f,getExprSecondOp(e));
    fprintf(f," )");
    break;
  case orExpr:
    fprintf(f,"( ");
    printSMVSimpleXExpr(f,getExprFirstOp(e));
    fprintf(f," | ");
    printSMVSimpleXExpr(f,getExprSecondOp(e));
    fprintf(f," )");
    break;
  case impliesExpr:
    fprintf(f,"( ");
    printSMVSimpleXExpr(f,getExprFirstOp(e));
    fprintf(f," -> ");
    printSMVSimpleXExpr(f,getExprSecondOp(e));
    fprintf(f," )");
    break;
  case iffExpr:
    fprintf(f,"( ");
    printSMVSimpleXExpr(f,getExprFirstOp(e));
    fprintf(f," <-> ");
    printSMVSimpleXExpr(f,getExprSecondOp(e));
    fprintf(f," )");
    break;
  case notExpr:
    fprintf(f,"( ! ");
    printSMVSimpleXExpr(f,getExprOp(e));
    fprintf(f," )");
    break;
  default:
    abort();
  }
}
int optPrintPropExpr(FILE * f, tExpr e) 
{
  if(isSMVPropExpr(e)) {
    fprintf(f,"INVAR ");
    printExpr(f,e);
    fprintf(f,"\n");
    return 1;
  }
  return 0;
}

int optPrintSimpleXExpr(FILE * f, tExpr e) 
{
  if(isSMVSimpleXExpr(e)) {
    fprintf(f,"TRANS ");
    printSMVSimpleXExpr(f,e);
    fprintf(f,"\n");
    return 1;
  }
  return 0;
}

void printSMVConstraint(FILE * f, tExpr e) 
{
  if (optPrintPropExpr(f,e)) return;
  if (optPrintSimpleXExpr(f,e)) return;
  {  
    static int ltln = 10000;
    char template_in[] = "/tmp/ltl2smv-inXXXXXX";
    char template_out[] = "/tmp/ltl2smv-outXXXXXX";
    char buf[100];
    int fd_in, fd_out;
    FILE * fin, * fout;

    if ((fd_in = mkstemp(template_in)) < 0) abort();
    if ((fd_out = mkstemp(template_out)) < 0) abort();

    fin = fdopen(fd_in,"w");
    printExpr(fin,mkGlobally(e));
    fclose(fin);
    sprintf(buf, "sh -c \"ltl2smv %d %s | grep -v MODULE\" > %s", ltln++, template_in, template_out);
    system(buf);
    fout = fdopen(fd_out,"r");
    { 
      int b;
      char buf[1024];
      while ((b = fread(buf,1,1024,fout)) > 0) {
	fwrite(buf,1,b,f);
      }
    }
    fclose(fout);
    remove(template_in);
    remove(template_out);
  }
}

void printSMV(FILE * f, tDom dom) 
{
  tElementSet entities;
  tPropSet globals;

  for (entities = getDomEntities(dom);
       !isemptyElementSet(entities); 
       entities = nextElementSet(entities)) {
    tElement entity = firstElement(entities);
    tAttrSet attrs = getElementAttrs(entity);

    fprintf(f,"MODULE %s\n", getNameString(getElementName(entity)));
    while(! isemptyAttrSet(attrs)) {
      tAttr attr = firstAttr(attrs);
      if (getSortKind(getAttrSort(attr)) == classSort) {
	tNameSet range = getElementRange(getDomElementByName(dom,getSortClassName(getAttrSort(attr))));
	fprintf(f,"  VAR %s : { ", 
		getNameString(getAttrName(attr)));
	while(! isemptyNameSet(range)) {
	  tName n = firstName(range);
	  fprintf(f,"%s", getNameString(n));
	  range = nextNameSet(range);
	  if (! isemptyNameSet(range)) {
	    fprintf(f,", ");
	  }
	}
	fprintf(f," };\n");
      } else if (getSortKind(getAttrSort(attr)) == intSort) {
	fprintf(f,"  VAR %s : -127..128;\n",
		getNameString(getAttrName(attr)));
      } else { /* boolSort */
	fprintf(f,"  VAR %s : %s;\n", 
		getNameString(getAttrName(attr)),
		getSortString(getAttrSort(attr)));
      }
      attrs = nextAttrSet(attrs);
    }
    fprintf(f,"\n");
  }

  fprintf(f,"MODULE main\n");
  for (entities = getDomEntities(dom);
       !isemptyElementSet(entities); 
       entities = nextElementSet(entities)) {
    tElement entity = firstElement(entities);
    tNameSet range = getElementRange(entity);

    while(! isemptyNameSet(range)) {
      tName n = firstName(range);
      fprintf(f,"  VAR %s : %s;\n", 
	     getNameString(n),
	     getNameString(getElementName(entity)));
      range = nextNameSet(range);
    }
    fprintf(f,"\n");
  }
  
  {  
    int first = 1;

    fprintf(f, "  VAR hidden : \{ ");  
    for (entities = getDomEntities(dom);
	 !isemptyElementSet(entities); 
	 entities = nextElementSet(entities)) {
      tElement entity = firstElement(entities);
      tNameSet range = getElementRange(entity);
      
      while(! isemptyNameSet(range)) {
	tName n = firstName(range);
	
	if (! first) fprintf(f, ", ");
	first = 0;
	
	fprintf(f,"%s", getNameString(n));
	range = nextNameSet(range);
      }
    }
    fprintf(f," };\n\n");
  }

  for (globals = getDomGlobalProps(dom);
       !isemptyPropSet(globals);
       globals = nextPropSet(globals)) {

    tProp prop = firstProp(globals);

    switch(getPropCategory(prop)) {
    case constraintCategory:
      fprintf(f,"-- CONSTRAINT ");
      printExpr(f,getPropExpr(prop));
      fprintf(f,"\n");
      fflush(stdout);
      printSMVConstraint(f, getPropExpr(prop));
      break;
    case assertionCategory:
      fprintf(f,"-- ASSERTION ");
      printExpr(f,getPropExpr(prop));
      fprintf(f,"\n");
      fprintf(f,"LTLSPEC ");
      printExpr(f,mkGlobally(getPropExpr(prop)));
      fprintf(f,"\n");
      break;
    case possibilityCategory:
      fprintf(f,"-- POSSIBILITY ");
      printExpr(f,getPropExpr(prop));
      fprintf(f,"\n");
      fprintf(f,"LTLSPEC ");
      printExpr(f,mkGlobally(mkNot(getPropExpr(prop))));
      fprintf(f,"\n");
      break;
    default:
      abort();
    }
    printf("\n");
  }
}



