/******************************************************************************

  FileName    [ft-to-il.c]

  PackageName [ft]

  Synopsis    [FT to IL translation routines]

  Author      [Marco Pistore] 

  Copyright   [Copyright (C) 2003 by University of Trento.

  T-Tool is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  T-Tool is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information on the T-Tool see <http://dit.unitn.it/~ft>
  or email to <ft@dit.unitn.it>. Please report bugs to <ft@dit.unitn.it>.]

******************************************************************************/

#include <malloc.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include "ft.h"

tProp constantAttr(tAttr attr, tElement class)
{
  return mkProp(mkConstraintCategory(),
		mkNoEvent(),
		mkNoOrigin(),
		mkForall(mkName("_c"),
			 getElementName(class),
			 mkForall(mkName("_v"),
				  getSortClassName(getAttrSort(attr)),
				  mkImplies(mkEqual(mkDotVar(mkVar(mkNameAtom(mkName("_c"))), mkNameAtom(getAttrName(attr))),
						    mkVar(mkNameAtom(mkName("_v")))),
					    mkNext(mkEqual(mkDotVar(mkVar(mkNameAtom(mkName("_c"))), mkNameAtom(getAttrName(attr))),
							   mkVar(mkNameAtom(mkName("_v")))))))));
}

tProp fulfilledForever(tElement class)
{
  return mkProp(mkConstraintCategory(),
		mkNoEvent(),
		mkNoOrigin(),
		mkForall(mkName("_c"),
			 getElementName(class),
			 mkImplies(mkDotVar(mkVar(mkNameAtom(mkName("_c"))), mkNameAtom(mkName("fulfilled"))),
				   mkNext(mkDotVar(mkVar(mkNameAtom(mkName("_c"))), mkNameAtom(mkName("fulfilled")))))));
}

tExpr convertExprAux(tExpr e, tName self, tNameSet bv)
{
  tExpr res;

  switch(getExprKind(e)) {
  case varExpr:
    switch(getAtomKind(getExprVar(e))) {
    case nameAtom:
      if (inNameSet(getAtomName(getExprVar(e)),bv)) {
	res = e;
      } else {
	assert(self);
	res = mkDotVar(mkVar(mkNameAtom(self)),getExprVar(e));
      }
      break;
    case selfAtom:
      assert(self);
      res = mkVar(mkNameAtom(self));
      break;
    case actorAtom:
      if (self) {
	res = mkDotVar(mkVar(mkNameAtom(self)),mkNameAtom(mkName("actor")));
      } else {
	res = mkVar(mkNameAtom(mkName("actor")));
      }
      break;
    case dependerAtom:
      if (self) {
	res = mkDotVar(mkVar(mkNameAtom(self)),mkNameAtom(mkName("depender")));
      } else {
	res = mkVar(mkNameAtom(mkName("depender")));
      }
      break;
    case dependeeAtom:
      if (self) {
	res = mkDotVar(mkVar(mkNameAtom(self)),mkNameAtom(mkName("dependee")));
      } else {
	res = mkVar(mkNameAtom(mkName("dependee")));
      }
      break;
    default:
      abort();
    }
    break;
  case dotVarExpr:
    switch(getAtomKind(getExprDotSuffix(e))) {
    case nameAtom:
      res = mkDotVar(convertExprAux(getExprDotPrefix(e),self,bv),getExprDotSuffix(e));
      break;
    case actorAtom:
      res = mkDotVar(convertExprAux(getExprDotPrefix(e),self,bv),
		     mkNameAtom(mkName("actor")));
      break;
    case dependerAtom:
      res = mkDotVar(convertExprAux(getExprDotPrefix(e),self,bv),
		     mkNameAtom(mkName("depender")));
      break;
    case dependeeAtom:
      res = mkDotVar(convertExprAux(getExprDotPrefix(e),self,bv),
		     mkNameAtom(mkName("dependee")));
      break;
    case selfAtom:
    default:
      abort();
    }
    break;
  case andExpr:
    res = mkAnd(convertExprAux(getExprFirstOp(e),self,bv),
		convertExprAux(getExprSecondOp(e),self,bv));
    break;
  case orExpr:
    res = mkOr(convertExprAux(getExprFirstOp(e),self,bv),
	       convertExprAux(getExprSecondOp(e),self,bv));
    break;
  case impliesExpr:
    res = mkImplies(convertExprAux(getExprFirstOp(e),self,bv),
		    convertExprAux(getExprSecondOp(e),self,bv));
    break;
  case iffExpr:
    res = mkIff(convertExprAux(getExprFirstOp(e),self,bv),
		convertExprAux(getExprSecondOp(e),self,bv));
    break;
  case untilExpr:
    res = mkUntil(convertExprAux(getExprFirstOp(e),self,bv),
		  convertExprAux(getExprSecondOp(e),self,bv));
    break;
  case sinceExpr:
    res = mkSince(convertExprAux(getExprFirstOp(e),self,bv),
		  convertExprAux(getExprSecondOp(e),self,bv));
    break;
  case equalExpr:
    res = mkEqual(convertExprAux(getExprFirstOp(e),self,bv),
		  convertExprAux(getExprSecondOp(e),self,bv));
    break;
  case notEqualExpr:
    res = mkNotEqual(convertExprAux(getExprFirstOp(e),self,bv),
		     convertExprAux(getExprSecondOp(e),self,bv));
    break;
  case notExpr:
    res = mkNot(convertExprAux(getExprOp(e),self,bv));
    break;
  case nextExpr:
    res = mkNext(convertExprAux(getExprOp(e),self,bv));
    break;
  case prevExpr:
    res = mkPrev(convertExprAux(getExprOp(e),self,bv));
    break;
  case finallyExpr:
    res = mkFinally(convertExprAux(getExprOp(e),self,bv));
    break;
  case pastExpr:
    res = mkPast(convertExprAux(getExprOp(e),self,bv));
    break;
  case globallyExpr:
    res = mkGlobally(convertExprAux(getExprOp(e),self,bv));
    break;
  case pastGloballyExpr:
    res = mkPastGlobally(convertExprAux(getExprOp(e),self,bv));
    break;
  case numberExpr:
    res = mkNumber(getExprNumber(e));
    break;
  case falseExpr:
    res = mkFalse();
    break;
  case trueExpr:
    res = mkTrue();
    break;
  case forallExpr:
    res = mkForall(getExprQuantVar(e),
		   getExprQuantClass(e),
		   convertExprAux(getExprQuantBody(e),self,
				  mkConsNameSet(getExprQuantVar(e),bv)));
    break;
  case existsExpr:
    res = mkExists(getExprQuantVar(e),
		   getExprQuantClass(e),
		   convertExprAux(getExprQuantBody(e),self,
				  mkConsNameSet(getExprQuantVar(e),bv)));
    break;
  case fulfilledExpr:
    res = mkDotVar(convertExprAux(getExprOp(e),self,bv), mkNameAtom(mkName("fulfilled")));
    break;
  case justFulfilledExpr:
    res = mkAnd(mkDotVar(convertExprAux(getExprOp(e),self,bv), mkNameAtom(mkName("fulfilled"))),
		mkNot(mkPrev(mkDotVar(convertExprAux(getExprOp(e),self,bv), mkNameAtom(mkName("fulfilled"))))));
    break;
  case justCreatedExpr:
    res = mkExists(mkName("_t"), 
		   getSortClassName(getExprType(getExprOp(e))),
		   mkAnd(mkEqual(mkVar(mkNameAtom(mkName("_t"))),
				 convertExprAux(getExprOp(e),self,bv)),
			 mkNot(mkPrev(mkEqual(mkVar(mkNameAtom(mkName("_t"))),
					      mkVar(mkNameAtom(mkName("_t"))))))));
    break;
  default:
    abort();
  }
  return res;
}

tExpr convertExpr(tExpr e, tName self) 
{
  return convertExprAux(e, self, mkNoNames());
}

tExpr quantifyConvertExpr(tCategory cat, tName className, tExpr cond, tExpr body)
{
  switch (cat) {
  case constraintCategory:
  case assertionCategory:
    if (getExprKind(cond) == trueExpr) {
      return convertExpr(mkForall(mkName("_c"), className, body),
			 mkName("_c"));
    } else {
      return convertExpr(mkForall(mkName("_c"), className, mkImplies(cond, body)),
			 mkName("_c"));
    }
  case possibilityCategory:
    if (getExprKind(cond) == trueExpr) {
      return convertExpr(mkExists(mkName("_c"), className, body),
			 mkName("_c"));
    } else {
      return convertExpr(mkExists(mkName("_c"), className, mkAnd(cond, body)),
			 mkName("_c"));
    }
  default:
    abort();
  }
}

#warning ELEMENTARY (BOOL, INT) FREE VARIABLES IN CREATION TRIGGER 
tExpr creationTriggerExpr(tProp create, tElement class)
{
  tNameSet fv = getExprFv(getPropExpr(create));
  tExpr tmp = mkTrue();  
  tAttrSet tmpAttrs = NULL;

  assert(getPropCategory(create) != possibilityCategory);
    
  tmpAttrs = getElementAttrs(class);
  if (getElementKind(class) == goalEl) {
      tmpAttrs = addAttribute(mkAttribute(mkName("actor"),
					  mkClassSort(getElementActor(class)),
					  mkNoFacet()),
			      tmpAttrs);
  }
  if (getElementKind(class) == depEl) {
      tmpAttrs = addAttribute(mkAttribute(mkName("depender"),
					  mkClassSort(getElementDepender(class)),
					  mkNoFacet()),
			      tmpAttrs);
      tmpAttrs = addAttribute(mkAttribute(mkName("dependee"),
					  mkClassSort(getElementDependee(class)),
					  mkNoFacet()),
			      tmpAttrs);
  }

  while (! isemptyAttrSet(tmpAttrs)) {
    tAttr attr = firstAttr(tmpAttrs);
    if (inNameSet(getAttrName(attr), fv)) {
      tExpr andEl = mkEqual(mkDotVar(mkVar(mkNameAtom(mkName("_c"))),
				     mkNameAtom(getAttrName(attr))),
			    mkVar(mkNameAtom(getAttrName(attr))));
      if (getExprKind(tmp) != trueExpr) {
	tmp = mkAnd(andEl, tmp);
      }
      else {
	tmp = andEl;
      }
    }
    tmpAttrs = nextAttrSet(tmpAttrs);
  }
  
  tmp = mkImplies(getPropExpr(create),mkExists(mkName("_c"), getElementName(class), tmp));
  
  tmpAttrs = getElementAttrs(class);
  if (getElementKind(class) == goalEl) {
      tmpAttrs = addAttribute(mkAttribute(mkName("actor"),
					  mkClassSort(getElementActor(class)),
					  mkNoFacet()),
			      tmpAttrs);
  }
  if (getElementKind(class) == depEl) {
      tmpAttrs = addAttribute(mkAttribute(mkName("depender"),
					  mkClassSort(getElementDepender(class)),
					  mkNoFacet()),
			      tmpAttrs);
      tmpAttrs = addAttribute(mkAttribute(mkName("dependee"),
					  mkClassSort(getElementDependee(class)),
					  mkNoFacet()),
			      tmpAttrs);
  }

  while (! isemptyAttrSet(tmpAttrs)) {
    tAttr attr = firstAttr(tmpAttrs);
    if (inNameSet(getAttrName(attr), fv)) {
      tmp = mkForall(getAttrName(attr), getSortClassName(getAttrSort(attr)), tmp);
    }
    tmpAttrs = nextAttrSet(tmpAttrs);
  }

  return convertExpr(tmp,NULL);
}

tProp creationProp(tProp create, tElement class)
{
  tExpr convExpr;
  tExpr justCreatedExpr = mkJustCreated(mkVar(mkSelfAtom()));
  setExprType(getExprOp(justCreatedExpr),mkClassSort(class));

  switch(getPropEvent(create)) {
  case conditionEvent:
    convExpr = quantifyConvertExpr(getPropCategory(create),
				   getElementName(class),
				   justCreatedExpr,
				   getPropExpr(create));
    break;
  case triggerEvent:
    convExpr = creationTriggerExpr(create, class);
    break;
  case definitionEvent:
    convExpr = mkAnd(quantifyConvertExpr(getPropCategory(create),
					 getElementName(class),
					 justCreatedExpr,
					 getPropExpr(create)),
		     creationTriggerExpr(create, class));
    break;
  default:
    abort();
  }
  return mkProp(getPropCategory(create),mkNoEvent(),mkNoOrigin(),convExpr);
}

tProp invariantProp(tProp invar, tElement class)
{
  tExpr convExpr;
  convExpr = quantifyConvertExpr(getPropCategory(invar),getElementName(class),mkTrue(),getPropExpr(invar));
  return mkProp(getPropCategory(invar),mkNoEvent(),mkNoOrigin(),convExpr);
}

tProp fulfillmentProp(tProp fulfill, tElement class)
{
  tExpr convExpr;
  switch (getElementMode(class)) {
  case achieveMode:
    switch(getPropEvent(fulfill)) {
    case conditionEvent:
      convExpr = quantifyConvertExpr(getPropCategory(fulfill),
				     getElementName(class),
				     mkJustFulfilled(mkVar(mkSelfAtom())), 
				     getPropExpr(fulfill));
      break;
    case triggerEvent:
      convExpr = quantifyConvertExpr(getPropCategory(fulfill),
				     getElementName(class),
				     getPropExpr(fulfill),
				     mkFulfilled(mkVar(mkSelfAtom())));
      break;
    case definitionEvent:
      convExpr = mkAnd(quantifyConvertExpr(getPropCategory(fulfill),
					   getElementName(class),
					   mkJustFulfilled(mkVar(mkSelfAtom())), 
					   getPropExpr(fulfill)),
		       quantifyConvertExpr(getPropCategory(fulfill),
					   getElementName(class),
					   getPropExpr(fulfill),
					   mkFulfilled(mkVar(mkSelfAtom()))));
      break;
    default:
      abort();
    }
    break;
    
  case maintainMode:
    switch(getPropEvent(fulfill)) {
    case conditionEvent:
      convExpr = quantifyConvertExpr(getPropCategory(fulfill),
				     getElementName(class),
				     mkFulfilled(mkVar(mkSelfAtom())), 
				     mkAnd(mkGlobally(getPropExpr(fulfill)),mkPastGlobally(getPropExpr(fulfill))));
      break;
    case triggerEvent:
      convExpr = quantifyConvertExpr(getPropCategory(fulfill),
				     getElementName(class),
				     mkAnd(mkGlobally(getPropExpr(fulfill)),mkPastGlobally(getPropExpr(fulfill))),
				     mkFulfilled(mkVar(mkSelfAtom())));
      break;
    case definitionEvent:
      convExpr = mkAnd(quantifyConvertExpr(getPropCategory(fulfill),
					   getElementName(class),
					   mkFulfilled(mkVar(mkSelfAtom())), 
					   mkAnd(mkGlobally(getPropExpr(fulfill)),mkPastGlobally(getPropExpr(fulfill)))),
		       quantifyConvertExpr(getPropCategory(fulfill),
					   getElementName(class),
					   mkAnd(mkGlobally(getPropExpr(fulfill)),mkPastGlobally(getPropExpr(fulfill))),
					   mkFulfilled(mkVar(mkSelfAtom()))));
      break;
    default:
      abort();
    }
    break;
    
  case achieveMaintainMode:
    switch(getPropEvent(fulfill)) {
    case conditionEvent:
      convExpr = quantifyConvertExpr(getPropCategory(fulfill),
				     getElementName(class),
				     mkFulfilled(mkVar(mkSelfAtom())), 
				     mkGlobally(getPropExpr(fulfill)));
      break;
    case triggerEvent:
      convExpr = quantifyConvertExpr(getPropCategory(fulfill),
				     getElementName(class),
				     mkGlobally(getPropExpr(fulfill)),
				     mkFulfilled(mkVar(mkSelfAtom())));
      break;
    case definitionEvent:
      convExpr = mkAnd(quantifyConvertExpr(getPropCategory(fulfill),
					   getElementName(class),
					   mkFulfilled(mkVar(mkSelfAtom())), 
					   mkGlobally(getPropExpr(fulfill))),
		       quantifyConvertExpr(getPropCategory(fulfill),
					   getElementName(class),
					   mkGlobally(getPropExpr(fulfill)),
					   mkFulfilled(mkVar(mkSelfAtom()))));
      break;
    default:
      abort();
    }
    break;


  case avoidMode:
    switch(getPropEvent(fulfill)) {
    case conditionEvent:
      convExpr = quantifyConvertExpr(getPropCategory(fulfill),
				     getElementName(class),
				     mkFulfilled(mkVar(mkSelfAtom())), 
				     mkAnd(mkGlobally(mkNot(getPropExpr(fulfill))),mkPastGlobally(mkNot(getPropExpr(fulfill)))));
      break;
    case triggerEvent:
      convExpr = quantifyConvertExpr(getPropCategory(fulfill),
				     getElementName(class),
				     mkAnd(mkGlobally(mkNot(getPropExpr(fulfill))),mkPastGlobally(mkNot(getPropExpr(fulfill)))),
				     mkFulfilled(mkVar(mkSelfAtom())));
      break;
    case definitionEvent:
      convExpr = mkAnd(quantifyConvertExpr(getPropCategory(fulfill),
					   getElementName(class),
					   mkFulfilled(mkVar(mkSelfAtom())), 
					   mkAnd(mkGlobally(mkNot(getPropExpr(fulfill))),mkPastGlobally(mkNot(getPropExpr(fulfill))))),
		       quantifyConvertExpr(getPropCategory(fulfill),
					   getElementName(class),
					   mkAnd(mkGlobally(mkNot(getPropExpr(fulfill))),mkPastGlobally(mkNot(getPropExpr(fulfill)))),
					   mkFulfilled(mkVar(mkSelfAtom()))));
      break;
    default:
      abort();
    }
    break;
  default: 
    abort();
  }

  return mkProp(getPropCategory(fulfill),mkNoEvent(),mkNoOrigin(),convExpr);
}

tDom FT2IL(tDom domFT) 
{
  tDom domIL = mkEmptyDom();

  tElementSet els = getDomElements(domFT);
  tPropSet props = getDomGlobalProps(domFT);
  
  while (! isemptyElementSet(els)) {
    tElement el = firstElement(els);

    tAttrSet attrs = getElementAttrs(el);
    tPropSet creats = getElementCreates(el);
    tPropSet invars = getElementInvars(el);
    tPropSet fulfills = getElementFulfills(el);

    tAttrSet attrsIL = mkNoAttributes();

    while (! isemptyAttrSet(attrs)) {
      tAttr attr = firstAttr(attrs);
      tAttr attrIL = mkAttribute(getAttrName(attr),
				 getAttrSort(attr),
				 mkNoFacet());
      attrsIL = addAttribute(attrIL,attrsIL);
      if (isConstFacet(getAttrFacet(attr))) {
	domIL = addGlobalProp(domIL, constantAttr(attrIL, el));
      }
      attrs = nextAttrSet(attrs);
    }
    
    if (getElementKind(el) == goalEl) {
      tAttr attrIL;

      attrIL = mkAttribute(mkName("fulfilled"),
			   mkBooleanSort(),
			   mkNoFacet());
      attrsIL = addAttribute(attrIL,attrsIL);
      domIL = addGlobalProp(domIL, fulfilledForever(el));

      attrIL = mkAttribute(mkName("actor"),
			   mkClassSort(getElementActor(el)),
			   mkNoFacet());
      attrsIL = addAttribute(attrIL,attrsIL);
      domIL = addGlobalProp(domIL, constantAttr(attrIL, el));
    }

    if (getElementKind(el) == depEl) {
      tAttr attrIL;

      attrIL = mkAttribute(mkName("fulfilled"),
			   mkBooleanSort(),
			   mkNoFacet());
      attrsIL = addAttribute(attrIL,attrsIL);
      domIL = addGlobalProp(domIL, fulfilledForever(el));

      attrIL = mkAttribute(mkName("depender"),
			   mkClassSort(getElementDepender(el)),
			   mkNoFacet());
      attrsIL = addAttribute(attrIL,attrsIL);
      domIL = addGlobalProp(domIL, constantAttr(attrIL, el));

      attrIL = mkAttribute(mkName("dependee"),
			   mkClassSort(getElementDependee(el)),
			   mkNoFacet());
      attrsIL = addAttribute(attrIL,attrsIL);
      domIL = addGlobalProp(domIL, constantAttr(attrIL, el));
    }

    domIL = addElement(domIL, mkClassEl(getElementName(el), attrsIL));

    while(! isemptyPropSet(creats)) {
      tProp create = firstProp(creats);
      domIL = addGlobalProp(domIL,creationProp(create, el));
      creats = nextPropSet(creats);
    }

    while(! isemptyPropSet(invars)) {
      tProp invar = firstProp(invars);
      domIL = addGlobalProp(domIL,invariantProp(invar, el));
      invars = nextPropSet(invars);
    }

    while(! isemptyPropSet(fulfills)) {
      tProp fulfill = firstProp(fulfills);
      domIL = addGlobalProp(domIL,fulfillmentProp(fulfill, el));
      fulfills = nextPropSet(fulfills);
    }

    els = nextElementSet(els);
  }

  while(! isemptyPropSet(props)) {
    tProp prop = firstProp(props);
    tProp propIL = mkProp(getPropCategory(prop),
			  mkNoEvent(),
			  mkNoOrigin(),
			  convertExpr(getPropExpr(prop), NULL));
    domIL = addGlobalProp(domIL,propIL);
    props = nextPropSet(props);
  }

  return domIL;
}
