/******************************************************************************

  FileName    [ft-type.c]

  PackageName [ft]

  Synopsis    [The type checking routines of the ft package]

  Author      [Marco Pistore] 

  Copyright   [Copyright (C) 2003 by University of Trento.

  T-Tool is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  T-Tool is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information on the T-Tool see <http://dit.unitn.it/~ft>
  or email to <ft@dit.unitn.it>. Please report bugs to <ft@dit.unitn.it>.]

******************************************************************************/

#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "ft.h"

#define ALLOC(type) (type *)(malloc(sizeof(type)))

int checkAttrs(tDom dom, tAttrSet attrs)
{
  int errors = 0;

  while(! isemptyAttrSet(attrs)) {
    tAttr attr = firstAttr(attrs);
    tAttrSet attrs2 = nextAttrSet(attrs);
    while (! isemptyAttrSet(attrs2)) {
      tAttr attr2 = firstAttr(attrs2);
      if(isequalName(getAttrName(attr),getAttrName(attr2))) {
	printf ("ERROR: redefining attribute %s...\n", 
		getNameString(getAttrName(attr)));
	errors = 1;
	break;
      }
      attrs2 = nextAttrSet(attrs2);
    }

    {
      tSort sort = getAttrSort(attr);
      
      if (getSortKind(sort) == classSort) {
	tElement el = getDomElementByName(dom, getSortClassName(sort));
	
	if(!el) {
	  printf("Warning: wrong type %s for attribute %s\n",
		 getNameString(getSortClassName(sort)),
		 getNameString(getAttrName(attr)));
	  errors = 1;
	} else {
	  setSortClass(sort, el);
	}
      }
    }

    attrs = nextAttrSet(attrs);
  }

  return errors;
}

int checkNames(tDom dom) 
{
  int errors = 0;

  tElementSet els = getDomElements(dom);
  while(! isemptyElementSet(els)) {
    tElement el = firstElement(els);
    tElementSet els2 = nextElementSet(els);
    while (! isemptyElementSet(els2)) {
      tElement el2 = firstElement(els2);
      if(isequalName(getElementName(el),getElementName(el2))) {
	printf ("ERROR: redefining element %s...\n", 
		getNameString(getElementName(el)));
	errors = 1;
	break;
      }
      els2 = nextElementSet(els2);
    }

    errors = errors || checkAttrs(dom, getElementAttrs(el));

    if(getElementKind(el) == goalEl) {
      tElement actor = getDomElementByName(dom, getElementActorName(el));
	
      if(!actor) {
	printf("Warning: wrong type %s for attribute actor\n",
	       getNameString(getElementActorName(el)));
	errors = 1;
      } else {
	setElementActor(el,actor);
      }
    }

    if(getElementKind(el) == depEl) {
      tElement depender = getDomElementByName(dom, getElementDependerName(el));
      tElement dependee = getDomElementByName(dom, getElementDependeeName(el));
	
      if(!depender) {
	printf("Warning: wrong type %s for attribute depender\n",
	       getNameString(getElementDependerName(el)));
	errors = 1;
      } else {
	setElementDepender(el,depender);
      }

      if(!dependee) {
	printf("Warning: wrong type %s for attribute dependee\n",
	       getNameString(getElementDependeeName(el)));
	errors = 1;
      } else {
	setElementDependee(el,dependee);
      }
    }

    els = nextElementSet(els);
  }

  return errors;
}

int checkAtom(tAtom a, tExpr expr, tElement el, tAttrSet attrs)
{
  int errors = 0;

  switch(getAtomKind(a)) {
  case nameAtom:
    {
      tAttrSet tmp = attrs;
      while (! isemptyAttrSet(tmp)) {
	tAttr attr = firstAttr(tmp);
	if (isequalName(getAtomName(a),	getAttrName(attr))) {
	  setExprType(expr, getAttrSort(attr));
	  break;
	}
	tmp = nextAttrSet(tmp);
      }
      if (isemptyAttrSet(tmp)) {
	printf("attribute %s not found...\n",
	       getNameString(getAtomName(a)));
	errors = 1;
      }
    }
    break;
  case selfAtom:
    if(el == NULL) {
      printf("self used outside the scope of an element...\n");
      errors = 1;
      break;
    }
    setExprType(expr, mkClassSort(el));
    break;
  case actorAtom:
    if(el == NULL) {
      printf("actor used outside the scope of an element...\n");
      errors = 1;
      break;
    }
    if (getElementKind(el) != goalEl) {
      printf("actor component only defined for goal elements...\n");
      errors = 1;
      break;
    }
    setExprType(expr, mkClassSort(getElementActor(el)));
    break;
  case dependerAtom:
    if(el == NULL) {
      printf("depender used outside the scope of an element...\n");
      errors = 1;
      break;
    }
    if (getElementKind(el) != depEl) {
      printf("depender component only defined for dependency elements...\n");
      errors = 1;
      break;
    }
    setExprType(expr, mkClassSort(getElementDepender(el)));
    break;
  case dependeeAtom:
    if(el == NULL) {
      printf("dependee used outside the scope of an element...\n");
      errors = 1;
      break;
    }
    if (getElementKind(el) != depEl) {
      printf("dependee component only defined for dependency elements...\n");
      errors = 1;
      break;
    }
    setExprType(expr, mkClassSort(getElementDependee(el)));
    break;
  default:
    abort();
  }

  return errors;
}

int checkExpr(tDom dom, tExpr expr, tElement el, tAttrSet attrs);

int checkBExpr(tDom dom, tExpr expr, tElement el, tAttrSet attrs)
{
  if (checkExpr(dom,expr,el,attrs)) return 1;
  if (getSortKind(getExprType(expr)) != boolSort) {
    printf("non-boolean value in boolean expression...\n");
    return 1;
  }
  return 0;
}

int checkExpr(tDom dom, tExpr expr, tElement el, tAttrSet attrs)
{
  int errors = 0;

  switch(getExprKind(expr)) {
  case varExpr:
    errors = errors || checkAtom(getExprVar(expr), expr, el, attrs);
    break;
  case dotVarExpr:
    if (checkExpr(dom,getExprDotPrefix(expr), el, attrs)) {
      errors = 1;
      break;
    }
    if (getSortKind(getExprType(getExprDotPrefix(expr))) != classSort) {
      printf("'.' applied to elementary type...\n");
      errors = 1;
      break;
    }
    if (getAtomKind(getExprDotSuffix(expr)) == selfAtom) {
      printf("self cannot be used after a '.'e...\n");
      errors = 1;
      break;
    }
    errors = errors || 
      checkAtom(getExprDotSuffix(expr), expr,
		getSortClass(getExprType(getExprDotPrefix(expr))),
		getElementAttrs(getSortClass(getExprType(getExprDotPrefix(expr)))));
    break;
  case andExpr:
  case orExpr:
  case impliesExpr:
  case iffExpr:
  case untilExpr:
  case sinceExpr:
    setExprType(expr, mkBooleanSort());
    errors = errors || checkBExpr(dom,getExprFirstOp(expr),el,attrs);
    errors = errors || checkBExpr(dom,getExprSecondOp(expr),el,attrs);
    break;
  case equalExpr:
  case notEqualExpr:
    setExprType(expr, mkBooleanSort());
    errors = errors || checkExpr(dom,getExprFirstOp(expr),el,attrs);
    errors = errors || checkExpr(dom,getExprSecondOp(expr),el,attrs);
    if(! isequalSort(getExprType(getExprFirstOp(expr)),
		     getExprType(getExprSecondOp(expr)))) {
      printf("Type error in operator = / != in expression...\n");
      errors = 1;
    }
    break;
  case notExpr:
  case nextExpr:
  case prevExpr:
  case finallyExpr:
  case pastExpr:
  case globallyExpr:
  case pastGloballyExpr:
    setExprType(expr, mkBooleanSort());
    errors = errors || checkBExpr(dom,getExprOp(expr),el,attrs);
    break;
  case numberExpr:
    setExprType(expr, mkIntegerSort());
    break;
  case falseExpr:
  case trueExpr:
    setExprType(expr, mkBooleanSort());
    break;
  case forallExpr:
  case existsExpr:
    setExprType(expr, mkBooleanSort());
    {
      tName qv = getExprQuantVar(expr);
      tName qc = getExprQuantClass(expr);
      tElement qe = getDomElementByName(dom,qc);

      if(! qe) {
	printf("Non existing element in quantifier sort %s...\n",
	       getNameString(qc));
	errors = 1;
      } else {
	tAttr newAttr = mkAttribute(qv,mkClassSort(qe),mkNoFacet());
	tAttrSet newAttrs = addAttribute(newAttr,attrs);

	setExprQuantType(expr, mkClassSort(qe));
	
	errors = errors || checkExpr(dom,getExprQuantBody(expr), el, newAttrs);
      }
    }
    break;
  case justFulfilledExpr:
  case fulfilledExpr:
    setExprType(expr, mkBooleanSort());
    errors = errors || checkExpr(dom,getExprOp(expr), el, attrs);
    if (getSortKind(getExprType(getExprOp(expr))) != classSort) {
      printf("fulfillment operator applied to elementary expression...\n");
      errors = 1;
      break;
    }
    if (getElementKind(getSortClass(getExprType(getExprOp(expr)))) != goalEl &&
	getElementKind(getSortClass(getExprType(getExprOp(expr)))) != depEl) {
      printf("fulfillment operator applied to non-intentional element...\n");
      errors = 1;
      break;
    }
    break;
  case justCreatedExpr:
    setExprType(expr, mkBooleanSort());
    errors = errors || checkExpr(dom,getExprOp(expr), el, attrs);
    if (getSortKind(getExprType(getExprOp(expr))) != classSort) {
      printf("creation operator applied to elementary expression...\n");
      errors = 1;
      break;
    }
    break;
  default:
    abort();
  }

  return errors;
}

int checkProperties(tDom dom)
{
  int errors = 0;

  tElementSet els = getDomElements(dom);
  while(! isemptyElementSet(els)) {
    tElement el = firstElement(els);
    
    {
      tPropSet creates = getElementCreates(el);
      while (! isemptyPropSet(creates)) {
	tProp prop = firstProp(creates);
	
	tCategory cat = getPropCategory(prop);
	tEvent ev = getPropEvent(prop);
	tOrigin orig = getPropOrigin(prop);
	tExpr expr = getPropExpr(prop);

	if(ev == noEvent) {
	  printf("Wrong event in creation property of element %s...\n",
		 getNameString(getElementName(el)));
	  errors = 1;
	}

	if((orig != noOrigin) && (getElementKind(el) != depEl)) {
	  printf("Origin specified in non-dependency element %s...\n",
		 getNameString(getElementName(el)));
	  errors = 1;
	}
	
	if((cat == possibilityCategory) && (ev == triggerEvent)) {
	  printf("Possibility triggers do not make sense in element %s...\n",
		 getNameString(getElementName(el)));
	  errors = 1;
	}

	errors = errors || checkExpr(dom,expr,el,getElementAttrs(el));

	creates = nextPropSet(creates);
      }
    }
    {     
      tPropSet invars = getElementInvars(el);
      while (! isemptyPropSet(invars)) {
	tProp prop = firstProp(invars);
	
	/*tCategory cat = getPropCategory(prop);*/
	tEvent ev = getPropEvent(prop);
	tOrigin orig = getPropOrigin(prop);
	tExpr expr = getPropExpr(prop);

	if(ev != noEvent) {
	  printf("Wrong event in invariant of element %s...\n",
		 getNameString(getElementName(el)));
	  errors = 1;
	}

	if((orig != noOrigin) && (getElementKind(el) != depEl)) {
	  printf("Origin specified in non-dependency element %s...\n",
		 getNameString(getElementName(el)));
	  errors = 1;
	}
	
	errors = errors || checkExpr(dom,expr,el,getElementAttrs(el));

	invars = nextPropSet(invars);
      }
    }
    if ((getElementKind(el) == goalEl) || (getElementKind(el) == depEl)) {

      tPropSet fulfills = getElementFulfills(el);

      while (! isemptyPropSet(fulfills)) {
	tProp prop = firstProp(fulfills);
	  
	tCategory cat = getPropCategory(prop);
	tEvent ev = getPropEvent(prop);
	tOrigin orig = getPropOrigin(prop);
	tExpr expr = getPropExpr(prop);
	
	if(ev == noEvent) {
	  printf("Wrong event in fulfillment property of %s...\n",
		 getNameString(getElementName(el)));
	  errors = 1;
	}
	
	if((orig != noOrigin) && (getElementKind(el) != depEl)) {
	  printf("Origin specified in non-dependency element %s...\n",
		 getNameString(getElementName(el)));
	  errors = 1;
	}
	
	if((cat == possibilityCategory) && (ev == triggerEvent)) {
	  printf("Possibility triggers do not make sense in element %s...\n",
		 getNameString(getElementName(el)));
	  errors = 1;
	}

	errors = errors || checkExpr(dom,expr,el,getElementAttrs(el));
	
	fulfills = nextPropSet(fulfills);
      }
    }
    
    els = nextElementSet(els);
  }

  {     
    tPropSet globals = getDomGlobalProps(dom);
    while (! isemptyPropSet(globals)) {
      tProp prop = firstProp(globals);
	
      /*tCategory cat = getPropCategory(prop);*/
      tEvent ev = getPropEvent(prop);
      tOrigin orig = getPropOrigin(prop);
      tExpr expr = getPropExpr(prop);

      if(ev != noEvent) {
	printf("Wrong event in global formula...\n");
	errors = 1;
      }

      if(orig != noOrigin) {
	printf("Origin specified in global formula...\n");
	errors = 1;
      }
	
      errors = errors || checkExpr(dom,expr,NULL,mkNoAttributes());

      globals = nextPropSet(globals);
    }
  }

  return errors;  
}

void typeCheckFT(tDom dom)
{
  int errors = 0;
  /* Check doubly defined names */
  errors = errors || checkNames(dom);

  /* Check properties and add types to expression nodes */
  errors = errors || checkProperties(dom);

  assert(!errors);
}

#warning TYPECHECKIL ACCEPTS FT-ONLY SYNTAX (EG JUSTFULFILLED)
void typeCheckIL(tDom dom)
{
  typeCheckFT(dom);
}
