%{
/******************************************************************************

  FileName    [il-gra.y]

  PackageName [ft]

  Synopsis    [The parser of the IL language]

  Author      [Marco Pistore] 

  Copyright   [Copyright (C) 2003 by University of Trento.

  T-Tool is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  T-Tool is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information on the T-Tool see <http://dit.unitn.it/~ft>
  or email to <ft@dit.unitn.it>. Please report bugs to <ft@dit.unitn.it>.]

******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include "ft.h"

#define YYERROR_VERBOSE

int il_error(char *s);
int il_lex();

static tDom dom;
tName curr_actor;

%}

%name-prefix="il_"

%union{
  int num;
  tName name;
  tElement class;
  tAttr attr;
  tAttrSet attrSet;
  tSort sort;
  tProp prop;
  tCategory category;
  tExpr expr;
}

%token<name> NAME
%token<num> NUMBER
%token CLASS INTEGER BOOLEAN CONSTRAINT ASSERTION POSSIBILITY 
%token TRUE FALSE FORALL EXISTS

%right IMPLIES
%left  IFF
%left  OR
%left  AND
%left  NOT
%left  UNTIL SINCE
%left  NEXT PREV GLOBALLY PAST_GLOBALLY FINALLY PAST
%left  EQUAL NOTEQUAL

%type<attrSet> attribute_list_opt
%type<attr> attribute
%type<sort> sort basic_sort
%type<prop> property
%type<category> property_category
%type<expr> temporal_formula expr var

%%

model	: class_list property_list_opt
	;

class_list
	: class
	| class class_list 
	;

class	: CLASS NAME attribute_list_opt
          {
            dom = addElement(dom,mkClassEl($2,$3));
          }
	;

attribute_list_opt
	: /* void */
          {
            $$ = mkNoAttributes();
          }
	| attribute attribute_list_opt
          { 
            $$ = addAttribute($1, $2); 
          }
	;

attribute
	: NAME ':' sort
          {
	    $$ = mkAttribute($1,$3,mkNoFacet());
          }
	;

sort	: NAME
          {
            $$ = mkClassNameSort($1);
          }
	| basic_sort
          {
            $$ = $1;
          }          
	;

basic_sort
	: INTEGER
          {
            $$ = mkIntegerSort();
          }
	| BOOLEAN
          {
            $$ = mkBooleanSort();
          }
	;

property_list_opt
        : /* void */
	| property 
	  {
	    dom = addGlobalProp(dom,$1);
	  }
          property_list_opt
	;

property
	: property_category temporal_formula
          {
	    $$ = mkProp($1,mkNoEvent(),mkNoOrigin(),$2);
          }
	;

property_category
	: CONSTRAINT
	  {
	    $$ = mkConstraintCategory();
          }
	| ASSERTION
	  {
	    $$ = mkAssertionCategory();
          }
	| POSSIBILITY
	  {
	    $$ = mkPossibilityCategory();
          }
	;

temporal_formula
	: expr
          {
	    $$ = $1;
          }
        ;

var     : NAME
          {
	    $$ = mkVar(mkNameAtom($1));
          }
        | var '.' NAME
          {
	    $$ = mkDotVar($1,mkNameAtom($3));
	  }
        ;

expr    : var
          {
            $$ = $1;
	  }
	| expr AND expr
          {
	    $$ = mkAnd($1,$3);
          }
	| expr OR expr
          {
	    $$ = mkOr($1,$3);
          }
	| expr IMPLIES expr
          {
	    $$ = mkImplies($1,$3);
          }
	| expr IFF expr
          {
	    $$ = mkIff($1,$3);
          }
	| NOT expr
          {
	    $$ = mkNot($2);
          }
	| NEXT expr
          {
	    $$ = mkNext($2);
          }
	| PREV expr
          {
	    $$ = mkPrev($2);
          }
	| expr UNTIL expr
	  {
            $$ = mkUntil($1,$3);
	  }
	| expr SINCE expr
	  {
            $$ = mkSince($1,$3);
	  }
	| GLOBALLY expr
          {
	    $$ = mkGlobally($2);
          }
	| PAST_GLOBALLY expr
          {
	    $$ = mkPastGlobally($2);
          }
	| FINALLY expr
          {
	    $$ = mkFinally($2);
          }
	| PAST expr
          {
	    $$ = mkPast($2);
          }
	| '(' expr ')'
          {
            $$ = $2;
          }
	| FORALL NAME ':' NAME '(' expr ')'
	  {
            $$ = mkForall($2,$4,$6);
	  }
	| EXISTS NAME ':' NAME '(' expr ')'
	  {
            $$ = mkExists($2,$4,$6);
	  }
	| NUMBER
	  {
            $$ = mkNumber($1);
	  }
	| FALSE
	  {
            $$ = mkFalse();
	  }
        | TRUE
	  {
            $$ = mkTrue();
	  }
	| expr EQUAL expr
	  {
            $$ = mkEqual($1,$3);
	  }
	| expr NOTEQUAL expr
	  {
            $$ = mkNotEqual($1,$3);
	  }
	;

%%

int il_error(char *s)
{
  extern int il_lineno;
  fprintf(stderr, "Error: %s at line %d\n", s, il_lineno);
  abort();
}

tDom parseIL(FILE * f)
{
  extern FILE * il_in;

  tDom parseDom;
  
  /* Using global dom variable for parsing */
  dom = mkEmptyDom();

  il_in = f;
  il_parse();
  
  parseDom = dom;
  dom = NULL;

  return parseDom;
}
