/******************************************************************************

  FileName    [il-to-smv.c]

  PackageName [ft]

  Synopsis    [IL to SMV translation routines]

  Author      [Marco Pistore] 

  Copyright   [Copyright (C) 2003 by University of Trento.

  T-Tool is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2 of the
  License, or (at your option) any later version.

  T-Tool is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information on the T-Tool see <http://dit.unitn.it/~ft>
  or email to <ft@dit.unitn.it>. Please report bugs to <ft@dit.unitn.it>.]

******************************************************************************/

#include <malloc.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include "ft.h"

tExpr expandQuantVarVal(tExpr e, tName var, tName val)
{
  tExpr res;
  
  switch(getExprKind(e)) {
  case varExpr:
    if(isequalName(var, getAtomName(getExprVar(e)))) {
      res = mkVar(mkNameAtom(val));
    } else {
      res = e;
    }
    break;
  case dotVarExpr:
    assert (getExprKind(getExprDotPrefix(e)) == varExpr);
    if(isequalName(var, getAtomName(getExprVar(getExprDotPrefix(e))))) {
      res = mkDotVar(mkVar(mkNameAtom(val)),getExprDotSuffix(e));
    } else {
      res = e;
    }
    break;
  case andExpr:
    res = mkAnd(expandQuantVarVal(getExprFirstOp(e),var,val),
		expandQuantVarVal(getExprSecondOp(e),var,val));
    break;
  case orExpr:
    res = mkOr(expandQuantVarVal(getExprFirstOp(e),var,val),
	       expandQuantVarVal(getExprSecondOp(e),var,val));
    break;
  case impliesExpr:
    res = mkImplies(expandQuantVarVal(getExprFirstOp(e),var,val),
		    expandQuantVarVal(getExprSecondOp(e),var,val));
    break;
  case iffExpr:
    res = mkIff(expandQuantVarVal(getExprFirstOp(e),var,val),
		expandQuantVarVal(getExprSecondOp(e),var,val));
    break;
  case untilExpr:
    res = mkUntil(expandQuantVarVal(getExprFirstOp(e),var,val),
		  expandQuantVarVal(getExprSecondOp(e),var,val));
    break;
  case sinceExpr:
    res = mkSince(expandQuantVarVal(getExprFirstOp(e),var,val),
		  expandQuantVarVal(getExprSecondOp(e),var,val));
    break;
  case equalExpr:
    res = mkEqual(expandQuantVarVal(getExprFirstOp(e),var,val),
		  expandQuantVarVal(getExprSecondOp(e),var,val));
    break;
  case notEqualExpr:
    res = mkNotEqual(expandQuantVarVal(getExprFirstOp(e),var,val),
		     expandQuantVarVal(getExprSecondOp(e),var,val));
    break;
  case notExpr:
    res = mkNot(expandQuantVarVal(getExprOp(e),var,val));
    break;
  case nextExpr:
    res = mkNext(expandQuantVarVal(getExprOp(e),var,val));
    break;
  case prevExpr:
    res = mkPrev(expandQuantVarVal(getExprOp(e),var,val));
    break;
  case finallyExpr:
    res = mkFinally(expandQuantVarVal(getExprOp(e),var,val));
    break;
  case pastExpr:
    res = mkPast(expandQuantVarVal(getExprOp(e),var,val));
    break;
  case globallyExpr:
    res = mkGlobally(expandQuantVarVal(getExprOp(e),var,val));
    break;
  case pastGloballyExpr:
    res = mkPastGlobally(expandQuantVarVal(getExprOp(e),var,val));
    break;
  case numberExpr:
  case falseExpr:
  case trueExpr:
    res = e;
    break;
  case forallExpr:
  case existsExpr:
  default:
    abort();
  }
  return res;
}

tExpr expandQuantVar(tExprKind k, tExpr e, tName var, tNameSet range)
{
  tExpr res;

  assert(k == forallExpr || k == existsExpr);

  if (k == forallExpr) {
    res = mkTrue();
  } else {
    res = mkFalse();
  }

  while(! isemptyNameSet(range)) {
    tName val = firstName(range);
    tExpr el = expandQuantVarVal(e, var, val);
    
    if (getExprKind(res) == trueExpr || getExprKind(res) == falseExpr) {
      res = el;
    } else if (k == forallExpr) {
      res = mkAnd(res, el);
    } else {
      res = mkOr(res, el);
    }

    range = nextNameSet(range);
  }

  return res;
}

tExpr expandQuantExpr(tExpr e, tDom dom)
{
  tExpr res;
  tNameSet range;
  
  switch(getExprKind(e)) {
  case varExpr:
  case dotVarExpr:
    res = e;
    break;
  case andExpr:
    res = mkAnd(expandQuantExpr(getExprFirstOp(e),dom),
		expandQuantExpr(getExprSecondOp(e),dom));
    break;
  case orExpr:
    res = mkOr(expandQuantExpr(getExprFirstOp(e),dom),
	       expandQuantExpr(getExprSecondOp(e),dom));
    break;
  case impliesExpr:
    res = mkImplies(expandQuantExpr(getExprFirstOp(e),dom),
		    expandQuantExpr(getExprSecondOp(e),dom));
    break;
  case iffExpr:
    res = mkIff(expandQuantExpr(getExprFirstOp(e),dom),
		expandQuantExpr(getExprSecondOp(e),dom));
    break;
  case untilExpr:
    res = mkUntil(expandQuantExpr(getExprFirstOp(e),dom),
		  expandQuantExpr(getExprSecondOp(e),dom));
    break;
  case sinceExpr:
    res = mkSince(expandQuantExpr(getExprFirstOp(e),dom),
		  expandQuantExpr(getExprSecondOp(e),dom));
    break;
  case equalExpr:
    res = mkEqual(expandQuantExpr(getExprFirstOp(e),dom),
		  expandQuantExpr(getExprSecondOp(e),dom));
    break;
  case notEqualExpr:
    res = mkNotEqual(expandQuantExpr(getExprFirstOp(e),dom),
		     expandQuantExpr(getExprSecondOp(e),dom));
    break;
  case notExpr:
    res = mkNot(expandQuantExpr(getExprOp(e),dom));
    break;
  case nextExpr:
    res = mkNext(expandQuantExpr(getExprOp(e),dom));
    break;
  case prevExpr:
    res = mkPrev(expandQuantExpr(getExprOp(e),dom));
    break;
  case finallyExpr:
    res = mkFinally(expandQuantExpr(getExprOp(e),dom));
    break;
  case pastExpr:
    res = mkPast(expandQuantExpr(getExprOp(e),dom));
    break;
  case globallyExpr:
    res = mkGlobally(expandQuantExpr(getExprOp(e),dom));
    break;
  case pastGloballyExpr:
    res = mkPastGlobally(expandQuantExpr(getExprOp(e),dom));
    break;
  case numberExpr:
  case falseExpr:
  case trueExpr:
    res = e;
    break;
  case forallExpr:
    range = getElementRange(getDomElementByName(dom,getExprQuantClass(e)));
    res = expandQuantVar(forallExpr,
			 expandQuantExpr(getExprQuantBody(e),dom),
			 getExprQuantVar(e),range);
    break;
  case existsExpr:
    range = getElementRange(getDomElementByName(dom,getExprQuantClass(e)));
    res = expandQuantVar(existsExpr,
			 expandQuantExpr(getExprQuantBody(e),dom),
			 getExprQuantVar(e),range);
    break;
  default:
    abort();
  }
  return res;
}

tDom expandQuantifiers(tDom dom)
{
  tDom domNew = mkEmptyDom();

  tElementSet els = getDomElements(dom);
  tPropSet props = getDomGlobalProps(dom);
  
  while (! isemptyElementSet(els)) {
    tElement el = firstElement(els);
    domNew = addElement(domNew, el);
    els = nextElementSet(els);
  }

  while (! isemptyPropSet(props)) {
    tProp prop = firstProp(props);

    domNew = addGlobalProp(domNew, 
			   mkProp(getPropCategory(prop),
				  mkNoEvent(),
				  mkNoOrigin(),
				  expandQuantExpr(getPropExpr(prop),dom)));

    props = nextPropSet(props);
  }
  
  return domNew;
}

tNameSet collectPEVAux(tExpr e, tName n)
{
  switch(getExprKind(e)) {
  case varExpr:
    return mkNoNames();
  case dotVarExpr:
    if (getExprKind(getExprDotPrefix(e)) == varExpr &&
	isequalName(n,getAtomName(getExprVar(getExprDotPrefix(e))))) {
      return mkConsNameSet(getAtomName(getExprDotSuffix(e)), mkNoNames());
    } else {
      return collectPEVAux(getExprDotPrefix(e),n);
    }
  default:
    abort();
  }
}

tNameSet collectPEV(tExpr e, tName n)
{
  switch(getExprKind(e)) {
  case varExpr:
    return mkNoNames();
  case dotVarExpr:
    return collectPEVAux(getExprDotPrefix(e),n);
  case andExpr:
  case orExpr:
  case impliesExpr:
  case iffExpr:
  case untilExpr:
  case sinceExpr:
  case equalExpr:
  case notEqualExpr:
    return mkCupNameSet(collectPEV(getExprFirstOp(e),n),
			collectPEV(getExprSecondOp(e),n));
  case notExpr:
  case nextExpr:
  case prevExpr:
  case finallyExpr:
  case pastExpr:
  case globallyExpr:
  case pastGloballyExpr:
    return collectPEV(getExprOp(e),n);
  case numberExpr:
  case falseExpr:
  case trueExpr:
    return mkNoNames();
  case forallExpr:
  case existsExpr:
    if (isequalName(n,getExprQuantVar(e))) {
      return mkNoNames();
    } else {
      return collectPEV(getExprQuantBody(e),n);
    }
  default:
    abort();
  }
}

tName newId(tName base)
{
  static int idx = 1;
  char * baseName = getNameString(base);
  char * name = malloc(strlen(baseName)+10+3);
  sprintf(name,"_%s_%d", baseName,idx++);
  return mkName(name);
}

tExpr pointifyExprAttr(tExpr e, tName className, tName attrName, tName idName)
{
  tExpr res;

  switch(getExprKind(e)) {
  case varExpr:
    res = e;
    break;
  case dotVarExpr:
    if (getExprKind(getExprDotPrefix(e)) == varExpr &&
	isequalName(className,getAtomName(getExprVar(getExprDotPrefix(e)))) &&
	isequalName(attrName, getAtomName(getExprDotSuffix(e)))) {
      res = mkVar(mkNameAtom(idName));
    } else {
      res = mkDotVar(pointifyExprAttr(getExprDotPrefix(e),className,attrName,idName),getExprDotSuffix(e));
    }
    break;
  case andExpr:
    res = mkAnd(pointifyExprAttr(getExprFirstOp(e),className,attrName,idName),
		pointifyExprAttr(getExprSecondOp(e),className,attrName,idName));
    break;
  case orExpr:
    res = mkOr(pointifyExprAttr(getExprFirstOp(e),className,attrName,idName),
	       pointifyExprAttr(getExprSecondOp(e),className,attrName,idName));
    break;
  case impliesExpr:
    res = mkImplies(pointifyExprAttr(getExprFirstOp(e),className,attrName,idName),
		    pointifyExprAttr(getExprSecondOp(e),className,attrName,idName));
    break;
  case iffExpr:
    res = mkIff(pointifyExprAttr(getExprFirstOp(e),className,attrName,idName),
		pointifyExprAttr(getExprSecondOp(e),className,attrName,idName));
    break;
  case untilExpr:
    res = mkUntil(pointifyExprAttr(getExprFirstOp(e),className,attrName,idName),
		  pointifyExprAttr(getExprSecondOp(e),className,attrName,idName));
    break;
  case sinceExpr:
    res = mkSince(pointifyExprAttr(getExprFirstOp(e),className,attrName,idName),
		  pointifyExprAttr(getExprSecondOp(e),className,attrName,idName));
    break;
  case equalExpr:
    res = mkEqual(pointifyExprAttr(getExprFirstOp(e),className,attrName,idName),
		  pointifyExprAttr(getExprSecondOp(e),className,attrName,idName));
    break;
  case notEqualExpr:
    res = mkNotEqual(pointifyExprAttr(getExprFirstOp(e),className,attrName,idName),
		     pointifyExprAttr(getExprSecondOp(e),className,attrName,idName));
    break;
  case notExpr:
    res = mkNot(pointifyExprAttr(getExprOp(e),className,attrName,idName));
    break;
  case nextExpr:
    res = mkNext(pointifyExprAttr(getExprOp(e),className,attrName,idName));
    break;
  case prevExpr:
    res = mkPrev(pointifyExprAttr(getExprOp(e),className,attrName,idName));
    break;
  case finallyExpr:
    res = mkFinally(pointifyExprAttr(getExprOp(e),className,attrName,idName));
    break;
  case pastExpr:
    res = mkPast(pointifyExprAttr(getExprOp(e),className,attrName,idName));
    break;
  case globallyExpr:
    res = mkGlobally(pointifyExprAttr(getExprOp(e),className,attrName,idName));
    break;
  case pastGloballyExpr:
    res = mkPastGlobally(pointifyExprAttr(getExprOp(e),className,attrName,idName));
    break;
  case numberExpr:
    res = mkNumber(getExprNumber(e));
    break;
  case falseExpr:
    res = mkFalse();
    break;
  case trueExpr:
    res = mkTrue();
    break;
  case forallExpr:
    res = mkForall(getExprQuantVar(e),
		   getExprQuantClass(e),
		   pointifyExprAttr(getExprQuantBody(e),className,attrName,idName));
    break;
  case existsExpr:
    res = mkExists(getExprQuantVar(e),
		   getExprQuantClass(e),
		   pointifyExprAttr(getExprQuantBody(e),className,attrName,idName));
    break;
  default:
    abort();
  }
  return res;
}

tExpr pointifyExprVar(tExpr e, tName n, tElement el)
{
  tNameSet pev = collectPEV(e,n);
  tAttrSet attrs = getElementAttrs(el);
  tExpr res = e;

  if (isemptyNameSet(pev)) return e;

  while(! isemptyAttrSet(attrs)) {
    tAttr attr = firstAttr(attrs);

    if (inNameSet(getAttrName(attr), pev) && getSortKind(getAttrSort(attr)) == classSort) {
      {
	tName id = newId(getAttrName(attr));
	res = mkForall(id,
		       getSortClassName(getAttrSort(attr)),
		       mkImplies(mkEqual(mkVar(mkNameAtom(id)), 
					 mkDotVar(mkVar(mkNameAtom(n)),mkNameAtom(getAttrName(attr)))),
				 pointifyExprAttr(res, n, getAttrName(attr),id)));
      }
    }

    attrs = nextAttrSet(attrs);
  }
   
  return res;
}

tExpr pointifyExpr(tExpr e, tDom dom)
{
  tExpr res, tmp;

  switch(getExprKind(e)) {
  case varExpr:
  case dotVarExpr:
    res = e;
    break;
  case andExpr:
    res = mkAnd(pointifyExpr(getExprFirstOp(e),dom),
		pointifyExpr(getExprSecondOp(e),dom));
    break;
  case orExpr:
    res = mkOr(pointifyExpr(getExprFirstOp(e),dom),
	       pointifyExpr(getExprSecondOp(e),dom));
    break;
  case impliesExpr:
    res = mkImplies(pointifyExpr(getExprFirstOp(e),dom),
		    pointifyExpr(getExprSecondOp(e),dom));
    break;
  case iffExpr:
    res = mkIff(pointifyExpr(getExprFirstOp(e),dom),
		pointifyExpr(getExprSecondOp(e),dom));
    break;
  case untilExpr:
    res = mkUntil(pointifyExpr(getExprFirstOp(e),dom),
		  pointifyExpr(getExprSecondOp(e),dom));
    break;
  case sinceExpr:
    res = mkSince(pointifyExpr(getExprFirstOp(e),dom),
		  pointifyExpr(getExprSecondOp(e),dom));
    break;
  case equalExpr:
    res = mkEqual(pointifyExpr(getExprFirstOp(e),dom),
		  pointifyExpr(getExprSecondOp(e),dom));
    break;
  case notEqualExpr:
    res = mkNotEqual(pointifyExpr(getExprFirstOp(e),dom),
		     pointifyExpr(getExprSecondOp(e),dom));
    break;
  case notExpr:
    res = mkNot(pointifyExpr(getExprOp(e),dom));
    break;
  case nextExpr:
    res = mkNext(pointifyExpr(getExprOp(e),dom));
    break;
  case prevExpr:
    res = mkPrev(pointifyExpr(getExprOp(e),dom));
    break;
  case finallyExpr:
    res = mkFinally(pointifyExpr(getExprOp(e),dom));
    break;
  case pastExpr:
    res = mkPast(pointifyExpr(getExprOp(e),dom));
    break;
  case globallyExpr:
    res = mkGlobally(pointifyExpr(getExprOp(e),dom));
    break;
  case pastGloballyExpr:
    res = mkPastGlobally(pointifyExpr(getExprOp(e),dom));
    break;
  case numberExpr:
  case trueExpr:
  case falseExpr:
    res = e;
    break;
  case forallExpr:
    tmp = pointifyExprVar(getExprQuantBody(e),getExprQuantVar(e),getDomElementByName(dom,getExprQuantClass(e)));
    res = mkForall(getExprQuantVar(e),
		   getExprQuantClass(e),
		   pointifyExpr(tmp,dom));
    break;
  case existsExpr:
    tmp = pointifyExprVar(getExprQuantBody(e),getExprQuantVar(e),getDomElementByName(dom,getExprQuantClass(e)));
    res = mkExists(getExprQuantVar(e),
		   getExprQuantClass(e),
		   pointifyExpr(tmp,dom));
    break;
  default:
    abort();
  }
  return res;
}

tDom pointify(tDom dom) 
{
  tDom domNew = mkEmptyDom();

  tElementSet els = getDomElements(dom);
  tPropSet props = getDomGlobalProps(dom);
  
  while (! isemptyElementSet(els)) {
    tElement el = firstElement(els);
    domNew = addElement(domNew, el);
    els = nextElementSet(els);
  }

  while (! isemptyPropSet(props)) {
    tProp prop = firstProp(props);

    domNew = addGlobalProp(domNew, 
			   mkProp(getPropCategory(prop),
				  mkNoEvent(),
				  mkNoOrigin(),
				  pointifyExpr(getPropExpr(prop),dom)));

    props = nextPropSet(props);
  }
  
  return domNew;
}

tExpr addExistsPastOp(tExpr e, tExprKind op) 
{
  tNameSet fn = getExprFv(e);
  tExpr exists = mkTrue();

  if (isemptyNameSet(fn)) return e;

  while (! isemptyNameSet(fn)) {
    tName n = firstName(fn);
      
    tExpr tmp = mkDotVar(mkVar(mkNameAtom(n)), mkNameAtom(mkName("exists")));
    if (getExprKind(exists) == trueExpr) {
      exists = tmp;
    } else {
      exists = mkAnd(exists,tmp);
    }
      
    fn = nextNameSet(fn);
  }
    
  assert(op == prevExpr || op == pastExpr || op == pastGloballyExpr);

  if (op == prevExpr || op == pastExpr) {
    return mkAnd(exists,e);
  } else {
    return mkImplies(exists,e);
  }
}

tExpr addExistsExpr(tExpr e)
{
  tExpr res;

  switch(getExprKind(e)) {
  case varExpr:
  case dotVarExpr:
    res = e;
    break;
  case andExpr:
    res = mkAnd(addExistsExpr(getExprFirstOp(e)),
		addExistsExpr(getExprSecondOp(e)));
    break;
  case orExpr:
    res = mkOr(addExistsExpr(getExprFirstOp(e)),
	       addExistsExpr(getExprSecondOp(e)));
    break;
  case impliesExpr:
    res = mkImplies(addExistsExpr(getExprFirstOp(e)),
		    addExistsExpr(getExprSecondOp(e)));
    break;
  case iffExpr:
    res = mkIff(addExistsExpr(getExprFirstOp(e)),
		addExistsExpr(getExprSecondOp(e)));
    break;
  case untilExpr:
    res = mkUntil(addExistsExpr(getExprFirstOp(e)),
		  addExistsExpr(getExprSecondOp(e)));
    break;
  case sinceExpr:
    res = mkSince(addExistsExpr(getExprFirstOp(e)),
		  addExistsExpr(getExprSecondOp(e)));
    break;
  case equalExpr:
    res = mkEqual(addExistsExpr(getExprFirstOp(e)),
		  addExistsExpr(getExprSecondOp(e)));
    break;
  case notEqualExpr:
    res = mkNotEqual(addExistsExpr(getExprFirstOp(e)),
		     addExistsExpr(getExprSecondOp(e)));
    break;
  case notExpr:
    res = mkNot(addExistsExpr(getExprOp(e)));
    break;
  case nextExpr:
    res = mkNext(addExistsExpr(getExprOp(e)));
    break;
  case prevExpr:
    res = mkPrev(addExistsPastOp(addExistsExpr(getExprOp(e)),
				 prevExpr));
    break;
  case finallyExpr:
    res = mkFinally(addExistsExpr(getExprOp(e)));
    break;
  case pastExpr:
    res = mkPast(addExistsPastOp(addExistsExpr(getExprOp(e)),
				 pastExpr));
    break;
  case globallyExpr:
    res = mkGlobally(addExistsExpr(getExprOp(e)));
    break;
  case pastGloballyExpr:
    res = mkPastGlobally(addExistsPastOp(addExistsExpr(getExprOp(e)),
					 pastGloballyExpr));
    break;
  case numberExpr:
    res = mkNumber(getExprNumber(e));
    break;
  case falseExpr:
    res = mkFalse();
    break;
  case trueExpr:
    res = mkTrue();
    break;
  case forallExpr:
    res = mkForall(getExprQuantVar(e),
		   getExprQuantClass(e),
		   mkImplies(mkDotVar(mkVar(mkNameAtom(getExprQuantVar(e))),
				      mkNameAtom(mkName("exists"))),
			     addExistsExpr(getExprQuantBody(e))));
    break;
  case existsExpr:
    res = mkExists(getExprQuantVar(e),
		   getExprQuantClass(e),
		   mkAnd(mkDotVar(mkVar(mkNameAtom(getExprQuantVar(e))),
				  mkNameAtom(mkName("exists"))),
			 addExistsExpr(getExprQuantBody(e))));
    break;
  default:
    abort();
  }
  return res;
}

tExpr existsForever(tName el)
{
  return mkForall(mkName("_c"), el,
		  mkImplies(mkDotVar(mkVar(mkNameAtom(mkName("_c"))),mkNameAtom(mkName("exists"))),
			    mkNext(mkDotVar(mkVar(mkNameAtom(mkName("_c"))),mkNameAtom(mkName("exists"))))));
}

tExpr mandatoryAttributes(tName el, tAttrSet attrs)
{
  tExpr res = mkTrue();

  while (! isemptyAttrSet(attrs)) {
    tAttr a = firstAttr(attrs);

    if (getSortKind(getAttrSort(a)) == classSort) {
      tExpr e = mkDotVar(mkDotVar(mkVar(mkNameAtom(mkName("_c"))),
				  mkNameAtom(getAttrName(a))),
			 mkNameAtom(mkName("exists")));
      
      if (getExprKind(res) == trueExpr) {
	res = e;
      } else {
	res = mkAnd(res, e);
      }
    }

    attrs = nextAttrSet(attrs);
  }

  if (getExprKind(res) == trueExpr) {
    return res;
  } else {
    return mkForall(mkName("_c"), el, 
		    mkImplies(mkDotVar(mkVar(mkNameAtom(mkName("_c"))),
				       mkNameAtom(mkName("exists"))),res));
  }
}

tDom addExists(tDom dom) 
{
  tDom domNew = mkEmptyDom();

  tElementSet els = getDomElements(dom);
  tPropSet props = getDomGlobalProps(dom);
  
  while (! isemptyElementSet(els)) {
    tElement el = firstElement(els);

    tAttrSet attrs = getElementAttrs(el);

    tAttrSet attrsNew = addAttribute(mkAttribute(mkName("exists"),
						 mkBooleanSort(),
						 mkNoFacet()),
				     attrs);

    tElement el_new = mkClassEl(getElementName(el), attrsNew);
    setElementRange(el_new, getElementRange(el));
    domNew = addElement(domNew, el_new);
    
    domNew = addGlobalProp(domNew, mkProp(mkConstraintCategory(),
					  mkNoEvent(),
					  mkNoOrigin(),
					  existsForever(getElementName(el))));

    while(! isemptyAttrSet(attrs)) {
      tAttr attr = firstAttr(attrs);
      if (getSortKind(getAttrSort(attr)) == classSort) {
	tNameSet range = getElementRange(getDomElementByName(dom,getSortClassName(getAttrSort(attr))));
	if (isemptyNameSet(range)) {
	  fprintf(stderr, "WARNING: empty range for attribute %s of %s.\n",
		  getNameString(getAttrName(attr)), 
		  getNameString(getElementName(el))); 
	}
      }	
      attrs = nextAttrSet(attrs);
    }
    
    { 
      tExpr ma = mandatoryAttributes(getElementName(el), getElementAttrs(el));
      if (getExprKind(ma) != trueExpr) {

	domNew = addGlobalProp(domNew, mkProp(mkConstraintCategory(),
					      mkNoEvent(),
					      mkNoOrigin(),
					      ma));
      }
    }

    els = nextElementSet(els);
  }

  while (! isemptyPropSet(props)) {
    tProp prop = firstProp(props);

    domNew = addGlobalProp(domNew, 
			   mkProp(getPropCategory(prop),
				  mkNoEvent(),
				  mkNoOrigin(),
				  addExistsExpr(getPropExpr(prop))));

    props = nextPropSet(props);
  }
  
  return domNew;
}

tDom IL2SMV(tDom domIL)
{
  tDom domSMV;

  domSMV = addExists(domIL);
  /* printIL(stdout, domSMV); */
    
  domSMV = pointify(domSMV);
  /* printIL(stdout, domSMV); */

  domSMV = expandQuantifiers(domSMV);
  /* printBIL(stdout, domSMV); */

  return domSMV;
}
