/**CFile***********************************************************************

  FileName    [simdllMain.c]

  PackageName [simdll]

  Synopsis    [A SAT solver using the SIM library.]

  Description [The purpose of this program is to let the user
               do hands-on experimentation with the SIM library.]
		
  SeeAlso     [sim]

  Author      [Armando Tacchella, Davide Zambonin]

  Copyright   [Copyright (c) 2000 by DIST - Universita' di Genova, Italia.  
               All Rights Reserved. This software is for educational
               purposes only.  Permission is given to academic
               institutions to use, copy, and modify this software and
               its documentation provided that this introductory
               message is not removed, that this software and its
               documentation is used for the institutions' internal
               research and educational purposes, and that no monies
               are exchanged. No guarantee is expressed or implied by
               the distribution of this code.  Permission to
               distribute this code is given to ITC-IRST,
               provided that the code is distributed as is.
               Send bug-reports and/or questions to: sim@mrg.dist.unige.it]

  Revision    [v. 1beta]


******************************************************************************/

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

#include "sim.h"


/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

#define SIMDLL_OK     0
#define SIMDLL_NOFILE 1
#define SIMDLL_ERROR  2

/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Variable********************************************************************
  Synopsis     [Values for the configuration parameters.]
  Description  [Values for the configuration parameters.]
  SeeAlso      []
******************************************************************************/
int parms[SIM_PARAM_NUM];

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static void HandleEvents(int sig);
static void DisplayHelp(char ** argv);
static char * ParseCmdLine(int argc, char ** argv, int * parms);

/**AutomaticEnd***************************************************************/

/**Function********************************************************************

  Synopsis    [Main program for simdll.]

  Description [Main program for simdll. Reads a file (supplied as
               command-line argument) and performs initialization of a DLL 
	       module. Reads command line options. Solves the problem 
	       according to the options and prints stats.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
int 
main(
  int     argc, 
  char ** argv)
{

  FILE * inFilePtr;

  int    res        = SIM_UNKNOWN;
  char * inFileName = NULL;

#ifdef HAVE_SIGNAL_H
  /* Install the event handler for generic/timeout SIM exceptions. */
  signal(SIGUSR1, HandleEvents);
  signal(SIGXCPU, HandleEvents); 
#endif


  /* Parsing *****************************************************************/
  /* Check the command line for non-emptyness. */
  if (argc < 2) {
    /* Display help and exit. */
    DisplayHelp(argv);
  } else {
    /* Parse the command line, get the parameters and the file name. */
    inFileName = ParseCmdLine(argc, argv, parms); 
  }
  /* Open input file. */
  if ((inFilePtr = fopen(inFileName, "r")) == NULL) {
    printf("Cannot open %s\n", inFileName);
    return (SIMDLL_NOFILE);
  }
  /* Initialize dll and parse the input file. */
  Sim_DllParseDimacs(inFilePtr, parms);
  /* Echo the settings if output is human oriented. */
  if (parms[SIM_VERBOSITY] > 0) {
    Sim_DllPrintParams();
  }


  /* Preprocessing************************************************************/
  if (parms[SIM_PPROC_STRENGTH] > 0) {
    /* Simplify the input form according to the preprocessing strength. */
    /*  res = Sim_DllPreprocess(dll);  */
    /* If the formula was trivially satisfiable or unsatisfiable 
       output result and exit.*/
    if (res != SIM_UNKNOWN) { 
      Sim_DllPrintResult(res);
      return (SIMDLL_OK);
    }
  }
  
  /* Searching ***************************************************************/

  res = Sim_DllSolve();

  /* Wrapping up *************************************************************/

  /* Output result, runtime, and statistics */
  Sim_DllPrintResult(res); 
  /* Shut down the solver. */
  Sim_DllClear(); 

  return (SIMDLL_OK);

} /* End of main. */


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [Event handler.]

  Description [Traps SIGXCPU and SIGUSR1 signals and allows stats to be
               printed, even when the solver fails to produce output.]

  SideEffects [none]

  SeeAlso     []

******************************************************************************/
static void 
HandleEvents(
  int sig)
{

#ifdef HAVE_SIGNAL_H
  /* Handle the event */
  switch (sig) {

  case SIGXCPU:
    Sim_DllPrintResult(SIM_TIME_FAIL);
    break;
  case SIGUSR1:
    Sim_DllPrintResult(SIM_GEN_FAIL);
    break;
  }

  exit (SIMDLL_OK);
#endif

}


/**Function********************************************************************

  Synopsis    [Help display.]

  Description [Describes command line parameters.]

  SideEffects [None]

  SeeAlso     []

******************************************************************************/
static void 
DisplayHelp(char ** argv)
{
  int i;

  printf("Usage: %s [options] <file>\n\n", argv[0]);
  printf("Options are:\n");
  printf("-timeout    <seconds>  0 means no timeout\n");
  printf("-memout     <Mbytes>   0 means no memout\n");
  printf("-heuristic  <heur>     see below\n");
  printf("-heurp      <n>        additional info for <heur>\n");
  printf("-numsol     <n>        1 or more DLL solutions\n");
  printf("-seed       <n>        0 hooks to the process ID\n");
  printf("-ppstrength <optlist>  DISABLED\n");
  printf("-model                 restrict splitting\n");
#ifdef LEARNING
  printf("-relevance  <n>        do relevance learning (order <n>)\n");
  printf("-size       <n>        do size learning (order <n>)\n");
#endif
  printf("-verbose               be verbose\n\n");
  printf("<heur> can be:\n");
  for (i = 0; i < SIM_HEUR_NUM - 1; i++) {
    printf("%s, ", Sim_heuristicName[i]);
  }
  printf("%s\n", Sim_heuristicName[SIM_HEUR_NUM - 1]);
  exit(0);

} /* End of DisplayHelp. */


/**Function********************************************************************

  Synopsis    [Command line parsing.]

  Description [Parses the command line and modifies sim behaviour
               according to the value of command line
	       parameters. Returns the input file name.]

  SideEffects [Loads the parms array with defaults. If any optional
               command line parameter is present, the default value is
	       updated with the current one.]

  SeeAlso     []

******************************************************************************/
static char *
ParseCmdLine(int argc, char ** argv, int * parms)
{

  int    i;

  char * inFileName = NULL;
  int    j = 0;

  /* Load default values into parms */
  for (i = 0; i < SIM_PARAM_NUM; i++) {
    parms[i] = SIM_ASK_DEFAULT;
  }

  i = 1;
  /* Explore the cmd line */
  while (i < argc) {
    if (argv[i][0] == '-') {
      if (strcmp(argv[i],"-timeout") == 0) {
	parms[SIM_TIMEOUT] = atoi(argv[i+1]);
	i += 2;
      } else if (strcmp(argv[i],"-memout") == 0) {
	parms[SIM_MEMOUT] = atoi(argv[i+1]) << 20;
	i += 2;
#ifdef LEARNING
      } else if (strcmp(argv[i],"-size") == 0) {
	parms[SIM_LEARN_TYPE] = SIM_SIZE;
	parms[SIM_LEARN_ORDER] = atoi(argv[i+1]);
	i += 2;
      } else if (strcmp(argv[i],"-relevance") == 0) {
	parms[SIM_LEARN_TYPE] = SIM_RELEVANCE;
	parms[SIM_LEARN_ORDER] = atoi(argv[i+1]);
	i += 2;
#endif
      } else if (strcmp(argv[i],"-numsol") == 0) {
	parms[SIM_SOL_NUM] = atoi(argv[i+1]);
	i += 2;
      } else if (strcmp(argv[i],"-heurp") == 0) {
	parms[SIM_HEUR_PARAM] = atoi(argv[i+1]);
	i += 2;
      } else if (strcmp(argv[i],"-seed") == 0) {
	parms[SIM_RND_SEED] = atoi(argv[i+1]);
	i += 2;
      } else if (strcmp(argv[i],"-verbose") == 0) {
	parms[SIM_VERBOSITY] = 1;
	i += 1;
      } else if (strcmp(argv[i],"-model") == 0) {
	parms[SIM_MODEL_PROPS] = 1;
	i += 1;
      } else if (strcmp(argv[i],"-heuristic") == 0) {
	for (j = 0; 
	     (j < SIM_HEUR_NUM) && 
	       (strcmp(argv[i + 1], Sim_heuristicName[j]) != 0);  
	     j++);
	if (j < SIM_HEUR_NUM) {
	  parms[SIM_HEURISTIC] = j;
	} else {
	  printf("Unrecognized heuristic %s\n",argv[i + 1]);
	}
	i += 2;
      } else if (strcmp(argv[i],"-ppstrength") == 0) {
	parms[SIM_PPROC_STRENGTH] = atoi(argv[i+1]);
	i += 2;
      } else {
	printf("Unrecognized parameter %s\n",argv[i]);
	i++;
      }
    } else {
      inFileName = argv[i++];
    }
  }

  return inFileName;

} /* End of ParseCmdLine. */

